package net.sf.distrib_rsa.cryptosystems.benaloh;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.util.Hashtable;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class BenalohKeySerializer {

	private static final long serialVersionUID = -2026488676222005449L;

	private static final int PUBLIC_KEY = 1;

	private static final int PRIVATE_KEY = 2;

	public static byte[] getSerialized(final BenalohKeyParameters key)
			throws IOException {
		final ByteArrayOutputStream baos = new ByteArrayOutputStream();
		final ObjectOutputStream oos = new ObjectOutputStream(baos);
		oos.writeLong(serialVersionUID);
		if (key instanceof BenalohPrivateKeyParameters) {
			oos.write(PRIVATE_KEY);
		} else {
			oos.write(PUBLIC_KEY);
		}

		oos.writeObject(key.getY());
		oos.writeObject(key.getMessageBorder());
		oos.writeObject(key.getModulus());

		if (key instanceof BenalohPrivateKeyParameters) {
			final BenalohPrivateKeyParameters priv = (BenalohPrivateKeyParameters) key;
			oos.writeObject(priv.getPQ()[0]);
			oos.writeObject(priv.getPQ()[1]);
			if (!key.getMessageBorder().equals(BigInteger.valueOf(2))) {
				// standard Benaloh key
				oos.writeObject(priv.getLookupList());
			}
		}

		oos.close();
		baos.flush();
		final byte[] retval = baos.toByteArray();
		baos.close();
		return retval;
	}

	public static BenalohKeyParameters deserialize(
			final byte[] naccacheSternKeyParameters) throws IOException,
			ClassNotFoundException {
		final ByteArrayInputStream bais = new ByteArrayInputStream(
				naccacheSternKeyParameters);
		final ObjectInputStream ois = new ObjectInputStream(bais);
		final long serializedData = ois.readLong();
		if (serializedData != serialVersionUID) {
			throw new IllegalArgumentException(
					"The supplied byte[] was not serialized by "
							+ BenalohKeySerializer.class.getName());
		}
		final int type = ois.read();
		final BigInteger y = (BigInteger) ois.readObject();
		final BigInteger r = (BigInteger) ois.readObject();
		final BigInteger n = (BigInteger) ois.readObject();
		BenalohKeyParameters key;
		if (type == PRIVATE_KEY) {
			final BigInteger p = (BigInteger) ois.readObject();
			final BigInteger q = (BigInteger) ois.readObject();
			if (!r.equals(BigInteger.valueOf(2))) {
				// Benaloh key
				final Hashtable lookup = (Hashtable) ois.readObject();
				key = new BenalohPrivateKeyParameters(y, r, n, p, q, lookup);
			} else {
				key = new BenalohPrivateKeyParameters(y, r, n, p, q);
			}
			ois.close();
			bais.close();
		} else {
			ois.close();
			bais.close();
			key = new BenalohKeyParameters(y, r, n);
		}
		return key;
	}

}
