/**
 * 
 */
package net.sf.distrib_rsa.protocols.computeD;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

import net.sf.distrib_rsa.InputStreamStatistics;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;


/**
 * @author lippold
 * 
 */
public class ProtocolVerifier {

	private static final Logger log = Logger.getLogger(ProtocolVerifier.class);

	/**
	 * @param args
	 * @throws IOException
	 * @throws NoSuchProviderException
	 * @throws NoSuchAlgorithmException
	 */
	public static void main(final String[] args)
			throws NoSuchAlgorithmException, NoSuchProviderException,
			IOException {
		initLog4J();
		final BigInteger p_a = new BigInteger("18398883532747294832");
		final BigInteger p_b = new BigInteger("26545796194971443987");
		final BigInteger q_a = new BigInteger("26423306523444172764");
		final BigInteger q_b = new BigInteger("20004216416848700543");
		final BigInteger rsaN = new BigInteger(
				"2086670149102777797370903854463789804433");
		final int certainty = 20;
		final SecureRandom rand = new SecureRandom();
		final PipedInputStream clientIn = new PipedInputStream();
		final PipedOutputStream clientOut = new PipedOutputStream();
		final PipedInputStream serverIn = new PipedInputStream(clientOut);
		final PipedOutputStream serverOut = new PipedOutputStream(clientIn);
		final ObjectOutputStream clientObjOut = new ObjectOutputStream(
				clientOut);
		final ObjectOutputStream serverObjOut = new ObjectOutputStream(
				serverOut);
		InputStreamStatistics clientStat = new InputStreamStatistics(clientIn);
		InputStreamStatistics serverStat = new InputStreamStatistics(serverIn);
		final ObjectInputStream clientObjIn = new ObjectInputStream(clientStat);
		final ObjectInputStream serverObjIn = new ObjectInputStream(serverStat);
		log.debug("setting up client");
		final ComputeKeyClient client = new ComputeKeyClient(p_a, q_a, rsaN,
				clientObjIn, clientObjOut, "", certainty, rand);
		log.debug("Client set up");
		log.debug("setting up server");
		final ComputeKeyServer server = new ComputeKeyServer(p_b, q_b, rsaN,
				serverObjIn, serverObjOut, "", certainty, rand);
		synchronized (client) {
			log.debug("Server set up");
			client.start();
			log.debug("Client started");
			server.start();
			log.debug("Server started");
			try {
				// We have to wait, because if this thread ends early, the
				// PipedStreams throw exceptions (they were created in this
				// thread). Furthermore, we can only verify everything after
				// client and server have finished computations.
				client.wait();
			} catch (final InterruptedException e) {
				log.fatal("Interrupted while waiting for client to finish", e);
			}
		}
		try {
			// Wait for server to finish
			Thread.sleep(1000);
		} catch (final InterruptedException e) {
		}
		int testFailed = 0;

		// Test individual phi(N)
		// Client phi = 0 mod 4
		final BigInteger phiA = client.getPhi();
		final BigInteger phiAcomp = BigInteger.ZERO.subtract(p_a).subtract(q_a);
		if (!phiA.equals(phiAcomp)) {
			log.fatal("phiA not correct");
			testFailed++;
		}
		// Server phi = 3 mod 4
		final BigInteger phiB = server.getPhi();
		final BigInteger phiBcomp = rsaN.subtract(p_b).subtract(q_b).add(
				BigInteger.ONE);
		if (!phiB.equals(phiBcomp)) {
			log.fatal("phiB not correct");
			testFailed++;
		}
		final BigInteger phi = phiAcomp.add(phiBcomp);
		if (!phi.equals(phiA.add(phiB))) {
			log.fatal("Phi is wrong");
			testFailed++;
		}

		// Test public exponent
		final BigInteger expA = client.getExp();
		final BigInteger expB = server.getExp();
		if (!expA.equals(expB)) {
			log.fatal("Exponents not the same");
			testFailed++;
		}

		// Test gcd(exp, phi) = 1
		final BigInteger gcd = expA.gcd(phi);
		if (!gcd.equals(BigInteger.ONE)) {
			log.fatal("gcd(exp, phi) != 1");
			testFailed++;
		}

		// Test multiplicative sharing of phi(N)
		final BigInteger phiAm = client.getMultShare_Phi_n_mod_e();
		final BigInteger phiBm = server.getMultShare_Phi_n_mod_e();
		final BigInteger phi_mod_exp = phiAm.multiply(phiBm).mod(expA);
		if (!phi_mod_exp.equals(phi.mod(expA))) {
			log.fatal("Phi mod e muliplicatively shared is not phi mod e");
			log.info("Server x: " + server.getX());
			log.info("Client y: " + client.getY());
			log.info("x + y: " + server.getX().add(client.getY()));
			log.info("x + y mod exp: "
					+ server.getX().add(client.getY()).mod(expA));
			log.info("Server a (= phi) " + server.getPhi());
			log.info("Client b (=r_e)  " + client.getZeta());
			log.info("a * b: " + server.getPhi().multiply(client.getZeta()));
			log.info("a * b mod exp: "
					+ server.getPhi().multiply(client.getZeta()).mod(expA));
			log.info("phiAm: " + phiAm);
			// log.info("phiAm: " + phiAm.toString(2));
			log.info("phiBm: " + phiBm);
			// log.info("phiBm: " + phiBm.toString(2));
			log.info("phi_mod_exp: " + phi_mod_exp);
			log.info("phi mod exp: " + phi.mod(expA));
			log.info("Exponent is: " + expA);
			// log.info("Exponent is: " + expA.toString(2));
			log.info("Exponent has " + expA.bitLength() + "bit");
			testFailed++;
		}

		// Test multiplicative sharing of zeta
		final BigInteger xiA = client.getZeta();
		final BigInteger xiB = server.getZeta();
		final BigInteger xiAB = xiA.multiply(xiB).mod(expA);
		final BigInteger phiInv = phi_mod_exp.modInverse(expA).negate().mod(
				expA);
		if (!xiAB.equals(phiInv)) {
			log.fatal("xiAB != phiInv");
			testFailed++;
		}

		// Test additive sharing of psi
		final BigInteger psiA = client.getPsi();
		final BigInteger psiB = server.getPsi();
		final BigInteger psiAB = psiA.add(psiB).mod(expA);
		if (!psiAB.equals(xiAB)) {
			log.fatal("xiA * xiB != psiA + psiB");
			log.info("client zeta: " + xiA);
			log.info("server zeta: " + xiB);
			log.info("client psi: " + psiA);
			log.info("server psi: " + psiB);
			testFailed++;
		}

		// Test alpha1 = psiA * phiB
		final BigInteger ring = server.getRing();
		final BigInteger alpha1A = client.getAlpha1();
		final BigInteger alpha1B = server.getAlpha1();
		final BigInteger alpha1 = alpha1A.add(alpha1B);
		final BigInteger psiAPhiB = psiA.multiply(phiB);
		if (!alpha1.mod(ring).equals(psiAPhiB)) {
			log.fatal("alpha1 is wrong.");
			log.info("psiA*phiB = " + psiAPhiB + " with "
					+ psiAPhiB.bitLength() + " bit");
			log.info("alpha1A is  " + alpha1A + " with " + alpha1A.bitLength()
					+ " bit");
			log.info("alpha1B is  " + alpha1B + " with " + alpha1B.bitLength()
					+ " bit");
			log.info("alpha1    = " + alpha1 + " with " + alpha1.bitLength()
					+ " bit");
			log.info("the ring used has " + ring.bitLength() + " bit");
			testFailed++;
		}

		// Test alpha2 = phiA * psiB
		final BigInteger alpha2A = client.getAlpha2();
		final BigInteger alpha2B = server.getAlpha2();
		final BigInteger alpha2 = alpha2A.add(alpha2B);
		final BigInteger phiAPsiB = phiA.multiply(psiB);
		if (!alpha2.mod(ring).equals(phiAPsiB.mod(ring))) {
			log.fatal("alpha2 is wrong.");
			log.info("phiA*psiB = " + phiAPsiB + " with "
					+ phiAPsiB.bitLength() + " bit");
			log.info("alpha2    = " + alpha2 + " with " + alpha2.bitLength()
					+ " bit");
			log.info("the ring used has " + ring.bitLength() + " bit");
			testFailed++;
		}

		// verify alpha = (psiA + psiB)*(phiA + phiB);
		final BigInteger phiAPsiA = phiA.multiply(psiA);
		final BigInteger phiBPsiB = phiB.multiply(psiB);
		final BigInteger alpha = alpha1.add(alpha2).add(phiAPsiA).add(phiBPsiB);
		final BigInteger alphaComp = phiAPsiA.add(phiBPsiB).add(phiAPsiB).add(
				psiAPhiB);
		if (!alpha.mod(ring).equals(alphaComp)) {
			log.fatal("alpha and alphaComp are not equal");
			log.info("alpha:     " + alpha + " with " + alpha.bitLength()
					+ " bit");
			log.info("alphaComp: " + alphaComp + " with "
					+ alphaComp.bitLength() + " bit");
			log.info("the ring used has " + ring.bitLength() + " bit");
		}

		// verify x - y = alpha
		final BigInteger x = server.getX();
		final BigInteger y = client.getY();
		final BigInteger x_y = x.subtract(y);
		if (!x_y.equals(alpha.mod(ring))) {
			log.fatal("x - y != alpha");
			log.info("x-y:   " + x_y);
			log.info("alpha: " + alpha);
		}

		// verify that d*e = 1 mod phi
		final BigInteger clientD = client.getD();
		final BigInteger serverD = server.getD();
		final BigInteger d = clientD.add(serverD);
		final BigInteger decrypt = d.multiply(expA);
		if (!decrypt.mod(phi).equals(BigInteger.ONE)) {
			log.fatal("d*e mod phi != 1 mod phi");
			log.info("d*e mod phi = " + decrypt.mod(phi) + " mod phi");
			log.info("phi is        " + phi);
			testFailed++;
		}

		if (testFailed > 0) {
			log.info(testFailed + " tests failed");
		} else {
			log.info("All tests successful");
		}
		log.info("Client received " + clientStat.getByteCount() + "");
		log.info("Server received " + serverStat.getByteCount() + "");
		
	}

	private static void initLog4J() {
		final Properties log4jproperties = new Properties();
		// configure two loggers: stdout (console) and R (file) logging
		log4jproperties.setProperty("log4j.rootLogger", "DEBUG, C, R");

		// configure A1 for console
		log4jproperties.setProperty("log4j.appender.C",
				"org.apache.log4j.ConsoleAppender");
		// configure custom Layout for console
		log4jproperties.setProperty("log4j.appender.C.layout",
				"org.apache.log4j.PatternLayout");
		log4jproperties.setProperty(
				"log4j.appender.C.layout.ConversionPattern",
				"[%-5p] [%d{HH:mm:ss}] %c.%M:%L %m\n");
		log4jproperties.setProperty("log4j.appender.C.threshold", "DEBUG");

		// configure file for file appending
		log4jproperties.setProperty("log4j.appender.R",
				"org.apache.log4j.RollingFileAppender");
		// set filename
		final SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
		log4jproperties.setProperty("log4j.appender.R.File",
				"ProtocolVerifier_" + df.format(new Date()) + ".log");
		// set max. filesize
		log4jproperties.setProperty("log4j.appender.R.MaxFileSize", "300KB");
		// set no. of backups
		log4jproperties.setProperty("log4j.appender.R.MaxBackupIndex", "3");
		// set custom logging layout
		log4jproperties.setProperty("log4j.appender.R.layout",
				"org.apache.log4j.PatternLayout");
		log4jproperties.setProperty(
				"log4j.appender.R.layout.ConversionPattern",
				"[%-5p] [%d{yyyy-MM-dd HH:mm:ss}] %m\n");
		// set custom logging level
		log4jproperties.setProperty("log4j.appender.R.threshold", "DEBUG");
		log4jproperties.setProperty(
				"log4j.logger.de.uni_bremen.informatik.lippold", "DEBUG");

		PropertyConfigurator.configure(log4jproperties);
	}

}
