/**
 * 
 */
package net.sf.distrib_rsa.protocols.gilboaSharing;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.cryptosystems.PrimeUtils;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohEngine;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohSystem;

import org.apache.log4j.Logger;
import org.bouncycastle.crypto.digests.SHA1Digest;


/**
 * @author lippold
 * 
 */
public class GilboaClient extends GilboaCommon {

	private static final Logger log = Logger.getLogger(GilboaClient.class);

	/**
	 * @uml.property name="nModPrimes"
	 * @uml.associationEnd multiplicity="(0 -1)"
	 *                     elementType="java.math.BigInteger"
	 */
	private final Vector nModPrimes = new Vector();

	/**
	 * @uml.property name="benalohSystems"
	 * @uml.associationEnd qualifier="prime:java.math.BigInteger
	 *                     de.uni_bremen.informatik.lippold.cryptosystems.benaloh.BenalohSystem"
	 */
	private final Hashtable benalohSystems = EnvironmentSetup
			.getBenalohSystems();

	/**
	 * @uml.property name="tries"
	 */
	private int tries = 0;

	public GilboaClient(final String remIP, final ObjectInputStream in,
			final ObjectOutputStream out) throws IOException,
			NoSuchAlgorithmException, NoSuchProviderException {
		super(remIP, in, out);
		nMod4 = BigInteger.ZERO;
	}

	public void run() {
		log.info(REMOTE_IP + "starting distributed sieving");
		sendDesiredPrimeLength();
		int command = -1;
		try {
			while (running && ((command = in.read()) != -1)) {

				switch (command) {

				case GilboaProtocol.PREFERRED_SHARED_KEY_SIZE:
					log.debug(REMOTE_IP + "Got PREFERRED_SHARED_KEY_SIZE");
					if (super.actualState == null) {
						log.debug(REMOTE_IP + "Got PREFERRED_SHARED_KEY_SIZE");
						super.actualState = states[0];
						readPreferredKeySize();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "PREFERRED_SHARED_KEY_SIZE");
					}
					break;

				case GilboaProtocol.START_BENALOH_SETUP:
					log.debug(REMOTE_IP + "Got START_BENALOH_SETUP");
					if (actualState.equals(states[0])) {
						actualState = states[2];
						sendBenalohSystems();
						log.debug(REMOTE_IP + "writing BENALOH_SETUP_FINISHED");
						out.write(GilboaProtocol.BENALOH_SETUP_FINISHED);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "START_BENALOH_SETUP");
					}
					break;

				case GilboaProtocol.BENALOH_SETUP_FINISHED:
					log.debug(REMOTE_IP + "Got BENALOH_SETUP_FINISHED");
					if (super.actualState.equals(states[2])) {
						log.debug(REMOTE_IP + "Got BENALOH_SETUP_FINISHED");
						super.actualState = states[3];
						log.debug(REMOTE_IP
								+ "Writing START_CANDIDATE_GENERATION");
						out.write(GilboaProtocol.START_CANDIDATE_GENERATION);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "BENALOH_SETUP");
					}
					break;

				case GilboaProtocol.BENALOH_SETUP_FAILED:
					log.debug(REMOTE_IP + "Got BENALOH_SETUP_FAILED");
					if (super.actualState.equals(states[2])) {
						log.fatal("Server reports that not all "
								+ "Benaloh systems were set up");
						log.fatal(REMOTE_IP + "Exiting");
						successful = false;
						endThread();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "BENALOH_SETUP_FAILED");
					}
					break;

				case GilboaProtocol.START_CANDIDATE_GENERATION:
					log.debug(REMOTE_IP + "Got START_CANDIDATE_GENERATION");
					if (super.actualState.equals(states[3])) {
						log.debug(REMOTE_IP + "Got START_CANDIDATE_GENERATION");
						startCandidateGeneration();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "START_CANDIDATE_GENERATION");
					}
					break;

				case GilboaProtocol.GENERATION_FINISHED:
					log.debug(REMOTE_IP + "Got GENERATION_FINISHED");
					if (super.actualState.equals(states[4])) {
						super.actualState = states[5];
						finishCandidateGeneration();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "GENERATION_FINISHED");
					}
					break;

				case GilboaProtocol.START_VERIFICATION:
					log.debug(REMOTE_IP + "Got START_VERIFICATION");
					if (super.actualState.equals(states[5])) {
						super.actualState = states[6];
						verifyN();
						if (super.actualState.equals(states[6])) {
							out.write(GilboaProtocol.VERIFICATION_FINISHED);
						}
						out.flush();
					}
					break;

				case GilboaProtocol.VERIFICATION_FINISHED:
					log.debug(REMOTE_IP + "Got VERIFICATION_FINISHED");
					if (super.actualState.equals(states[6])) {
						super.actualState = states[7];
						generateRsaN();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "VERIFICATION_FINISHED");
					}
					break;

				case GilboaProtocol.VERIFICATION_FAILED:
					log.debug(REMOTE_IP + "Got VERIFICATION_FAILED");
					if (super.actualState.equals(states[6])
							|| super.actualState.equals(states[9])) {
						log.fatal(REMOTE_IP + "Remote side doesn't think "
								+ "verification was successful,"
								+ " but we do.");
						log.fatal(REMOTE_IP + "Exiting.");
						successful = false;
						endThread();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "VERIFICATION_FAILED");
					}
					break;

				case GilboaProtocol.NEW_N:
					log.debug(REMOTE_IP + "Got NEW_N");
					if (super.actualState.equals(states[7])) {
						super.actualState = states[8];
						log.debug(REMOTE_IP + "Server accepted new RSA N. "
								+ "Now starting biprimality test.");
						out.write(GilboaProtocol.PREPARE_BIPRIME_TEST);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "NEW_N");
					}
					break;

				case GilboaProtocol.N_FAILED:
					log.debug(REMOTE_IP + "Got N_FAILED");
					if (super.actualState.equals(states[6])
							|| super.actualState.equals(states[5])) {
						log.fatal(REMOTE_IP + "Server reports that his table "
								+ "is incomplete although all "
								+ "primes are through");
						log.fatal(REMOTE_IP + "Exiting");
						successful = false;
						endThread();
					} else if (super.actualState.equals(states[8])) {
						log.info(REMOTE_IP + "Server detected that "
								+ "N has not enough bits");
						log.info(REMOTE_IP + "Restarting Generation");
						super.actualState = states[3];
						out.write(GilboaProtocol.START_CANDIDATE_GENERATION);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "N_FAILED");
					}
					break;

				case GilboaProtocol.PREPARE_BIPRIME_TEST:
					log.debug(REMOTE_IP + "Got PREPARE_BIPRIME_TEST");
					if (super.actualState.equals(states[8])) {
						super.actualState = states[9];
						if (biPrimalityTest()) {
							out.write(GilboaProtocol.START_RSA_KEY_GENERATION);
							out.flush();
						} else {
							log.info(REMOTE_IP + "Restarting generation");
							super.actualState = states[3];
							out
									.write(GilboaProtocol.START_CANDIDATE_GENERATION);
							out.flush();
						}

					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "PREPARE_BIPRIME_TEST");
					}
					break;

				case GilboaProtocol.START_RSA_KEY_GENERATION:
					log.debug("Got START_RSA_KEY_GENERATION");
					if (super.actualState.equals(states[9])) {
						log.info(REMOTE_IP + "Found RSA N: " + rsaN);
						log.debug(REMOTE_IP + "My p is " + p);
						log.debug(REMOTE_IP + "My q is " + q);
						log.info(REMOTE_IP + "Needed " + tries
								+ " tries to generate shared RSA N");
						endThread();
					}
					break;

				default:
					log.fatal(REMOTE_IP + PROTO_VIOL
							+ "Unknown command. Exiting.");
					successful = false;
					endThread();
					break;

				}
			}
		} catch (final Exception e) {
			log.error("Caught exception : ", e);
			successful = false;
		}
		log.debug(REMOTE_IP + "Last command was " + command);
		log.info(REMOTE_IP + "finished generation of RSA N");
		synchronized (this) {
			notifyAll();
		}
	}

	/**
	 * @throws IOException
	 * 
	 */
	private void generateRsaN() throws IOException {
		final Vector allPrimes = new Vector();
		allPrimes.addAll(genPrimes);
		allPrimes.addAll(verPrimes);
		rsaN = PrimeUtils.chineseRemainder(nModPrimes, allPrimes);
		allPrimes.clear();
		if (rsaN.bitLength() < super.preferredSharedKeySize) {
			log.info(REMOTE_IP + "Generated N is smaller than "
					+ preferredSharedKeySize + " bit");
			out.write(GilboaProtocol.N_FAILED);
			actualState = states[3];
			out.write(GilboaProtocol.START_CANDIDATE_GENERATION);
			out.flush();
		} else {
			log.info(REMOTE_IP + "New RSA n with " + rsaN.bitLength() + " bit");
			out.write(GilboaProtocol.NEW_N);
			out.writeObject(rsaN);
		}
	}

	/**
	 * @throws IOException
	 * @throws Exception
	 * @throws IOException
	 * 
	 */
	private void finishCandidateGeneration() throws IOException {
		setFinalPQ();
		out.write(GilboaProtocol.START_VERIFICATION);
		out.flush();
	}

	/**
	 * @throws Exception
	 * 
	 */
	private void startCandidateGeneration() throws Exception {
		tries++;
		super.actualState = states[4];
		super.populatePrimeLists();
		prepareNVector();
		if (generatePrimeCandidate()) {
			out.write(GilboaProtocol.GENERATION_FINISHED);
		}
		out.flush();
	}

	/**
	 * @throws IOException
	 * 
	 */
	private void readPreferredKeySize() throws IOException {
		final int serverRecommends = in.readInt();
		if (serverRecommends > preferredSharedKeySize) {
			log.fatal(REMOTE_IP + "Server wants bigger key size."
					+ " Not supported. Exiting.");
			endThread();
		} else {
			preferredSharedKeySize = serverRecommends;
			log
					.debug(REMOTE_IP + "agreed on keysize "
							+ preferredSharedKeySize);
			out.write(GilboaProtocol.START_BENALOH_SETUP);
			out.flush();
		}
	}

	/**
	 * Does a biPrimality test for the generated rsaN.
	 * 
	 * @return true If rsaN is composed of exactly two primes
	 * @throws Exception
	 *             If something unexpected happens
	 */
	private boolean biPrimalityTest() throws Exception {
		out.write(GilboaProtocol.PRIME_CERTAINTY);
		out.writeInt(super.primeCertainty);
		out.flush();
		if (in.read() == GilboaProtocol.PRIME_CERTAINTY) {
			final int remPrimeCert = in.readInt();
			super.primeCertainty = Math.max(primeCertainty, remPrimeCert);
		} else {
			final String msg = PROTO_VIOL + "expecting PRIME_CERTAINTY";
			log.fatal(REMOTE_IP + msg);
			throw new Exception(msg);
		}
		boolean retval = true;
		for (int i = 0; i < super.primeCertainty; i++) {
			if (sendWitness()) {
				log.debug((i + 1) + " / " + super.primeCertainty
						+ " tries passed.");
			} else {
				log.debug((i + 1) + " / " + super.primeCertainty
						+ " tries failed.");
				retval = false;
				break;
			}
		}
		return retval;
	}

	/**
	 * Tests if the rsaN is composed of exactly two primes. First, the
	 * candidates of both parties are fixed and then the test is performed by
	 * using the candidate that is the sum of both candidates.
	 * 
	 * @return True if the test is successful.
	 * @throws Exception
	 *             If the protocol is violated or the parties disagree about the
	 *             compositeness.
	 */
	private boolean sendWitness() throws Exception {
		// Inform the server that a BiPrime test starts
		out.write(GilboaProtocol.BIPRIME_TEST);
		BigInteger witness;
		// fix our half of the candidate
		witness = PrimeUtils.getRandom(rsaN, rand);
		SHA1Digest sha1 = new SHA1Digest();
		byte[] wData = witness.toByteArray();
		sha1.update(wData, 0, wData.length);
		byte[] digest = new byte[sha1.getDigestSize()];
		sha1.doFinal(digest, 0);
		// and inform server about the candidate but do not reveal it
		out.write(GilboaProtocol.SHA1_DIGEST);
		out.write(digest.length);
		out.write(digest);
		out.flush();
		// read the servers candidate
		if (in.read() == GilboaProtocol.SHA1_DIGEST) {
			byte[] remoteDigest = new byte[in.read()];
			in.readFully(remoteDigest);
			// reveal our candidate
			out.writeObject(witness);
			out.flush();
			// read the remote candidate and verify it
			BigInteger remCand = (BigInteger) in.readObject();
			wData = remCand.toByteArray();
			sha1.reset();
			sha1.update(wData, 0, wData.length);
			sha1.doFinal(digest, 0);
			if ((new BigInteger(1, remoteDigest)).equals(new BigInteger(1,
					digest))) {
				// compose the final candidate of both sharings
				witness = witness.add(remCand);
				while (PrimeUtils.jacobiSymbol(witness, rsaN) != 1) {
					witness = witness.add(BigInteger.ONE);
				}
			} else {
				String msg = REMOTE_IP + PROTO_VIOL + "SHA-1 digest "
						+ "of transferred object is not correct";
				log.fatal(msg);
				throw new Exception(msg);
			}
		} else {
			String msg = REMOTE_IP + PROTO_VIOL + "expected SHA1_DIGEST";
			log.fatal(msg);
			throw new Exception(msg);
		}

		// Do the simple test that catches almost all
		final BigInteger myExp = ((p.add(q)).negate()).divide(four);
		BigInteger jWitness = witness.modPow(myExp, rsaN);
		jWitness = jWitness.mod(rsaN);
		// log.debug(REMOTE_IP + "Sending Witness" + witness);
		out.write(GilboaProtocol.WITNESS);
		out.writeObject(witness);
		out.writeObject(jWitness);
		out.flush();
		BigInteger remWitness;
		if (in.read() == GilboaProtocol.WITNESS) {
			remWitness = (BigInteger) in.readObject();
		} else {
			final String msg = PROTO_VIOL + "expecting WITNESS";
			log.fatal(REMOTE_IP + msg);
			throw new Exception(msg);
		}
		jWitness = remWitness.multiply(jWitness);
		jWitness = jWitness.mod(rsaN);
		int retval = in.read();
		if ((jWitness.equals(BigInteger.ONE) || jWitness.equals(rsaN
				.subtract(BigInteger.ONE)))
				&& (retval == GilboaProtocol.N_VERIFIED)) {

			// If it succeeded, do a final test in the twisted group

			// First fix our share of the twisted candidate
			BigInteger[] twWitness = PrimeUtils.getRandomTwistedElement(rsaN,
					rand);
			sha1.reset();
			wData = twWitness[0].toByteArray();
			sha1.update(wData, 0, wData.length);
			wData = twWitness[1].toByteArray();
			sha1.update(wData, 0, wData.length);
			sha1.doFinal(digest, 0);

			// and inform the server about it

			out.write(GilboaProtocol.SHA1_DIGEST);
			out.write(digest.length);
			out.write(digest);
			out.flush();

			// read the servers hash value
			if (in.read() == GilboaProtocol.SHA1_DIGEST) {
				byte[] remoteDigest = new byte[in.read()];
				in.readFully(remoteDigest);

				// and reveal our element
				out.writeObject(twWitness[0]);
				out.writeObject(twWitness[1]);
				out.flush();

				// read the servers element
				BigInteger[] remCand = new BigInteger[2];
				remCand[0] = (BigInteger) in.readObject();
				remCand[1] = (BigInteger) in.readObject();

				// and verify it
				wData = remCand[0].toByteArray();
				sha1.reset();
				sha1.update(wData, 0, wData.length);
				wData = remCand[1].toByteArray();
				sha1.update(wData, 0, wData.length);
				sha1.doFinal(digest, 0);
				if ((new BigInteger(1, remoteDigest)).equals(new BigInteger(1,
						digest))) {

					// If all is correct, fix the final element
					twWitness = PrimeUtils.twistedAdd(twWitness, remCand);

				} else {
					String msg = REMOTE_IP + PROTO_VIOL + "SHA-1 digest "
							+ "of transferred object is not correct";
					log.fatal(msg);
					throw new Exception(msg);
				}
			} else {
				String msg = REMOTE_IP + PROTO_VIOL + "expected SHA1_DIGEST";
				log.fatal(msg);
				throw new Exception(msg);
			}

			// Do the clients part of the test in the twisted group
			BigInteger[] jTwWitness = PrimeUtils.twistedModPow(twWitness, p
					.add(q), rsaN);
			// and inform the server about it
			out.write(GilboaProtocol.TW_WITNESS);
			out.writeObject(twWitness[0]);
			out.writeObject(twWitness[1]);
			out.writeObject(jTwWitness[0]);
			out.writeObject(jTwWitness[1]);
			out.flush();

			// Read the servers results
			final BigInteger[] remTwWitness = new BigInteger[2];
			if (in.read() == GilboaProtocol.TW_WITNESS) {
				remTwWitness[0] = (BigInteger) in.readObject();
				remTwWitness[1] = (BigInteger) in.readObject();
			}

			// and combine the results to get the final result
			jTwWitness = PrimeUtils.twistedModMult(remTwWitness, jTwWitness,
					rsaN);

			retval = in.read();
			if (jTwWitness[1].equals(BigInteger.ZERO)) {
				if (retval == GilboaProtocol.N_VERIFIED) {
					// if both parties are sure that rsaN is prime for this
					// candidate, return true to the calling function
					log.info(REMOTE_IP + "For witness " + witness
							+ ", the rsaN passes the test");
					out.write(GilboaProtocol.N_VERIFIED);
					out.flush();
					return true;
				} else {
					// Throw exception on disagreement and exit
					final String msg = "Other side thinks that " + rsaN
							+ " is not prime, but I do";
					log.fatal(REMOTE_IP + msg);
					throw new Exception(msg);
				}
			} else {
				if (retval == GilboaProtocol.N_FAILED) {

					// Else return false if both parties agree that rsaN is not
					// composed of exactly two primes.
					log.info(REMOTE_IP + "N failed BiPrimality test");
					return false;
				} else {
					// Throw exception on disagreement and exit
					final String msg = "Other side thinks that " + rsaN
							+ " is prime, but I don't";
					log.fatal(REMOTE_IP + msg);
					throw new Exception(msg);
				}
			}
		} else {
			// Return false if the first test failed already
			log.info("RSA N failed first BiPrimality test. Restarting.");
			return false;
		}
	}

	/**
	 * Prepares the nModPrimes vector so that we can use Vector.set(int pos, Obj
	 * o).
	 */
	private void prepareNVector() {
		nModPrimes.clear();
		final Object o = new Object();
		while (nModPrimes.size() < genPrimes.size() + verPrimes.size()) {
			nModPrimes.add(o);
		}
	}

	private void sendDesiredPrimeLength() {
		try {
			log.debug(REMOTE_IP + "Writing PREFERRED_SHARED_KEY_SIZE");
			out.write(GilboaProtocol.PREFERRED_SHARED_KEY_SIZE);
			log.debug(REMOTE_IP + "Sending Desired Prime Length of "
					+ preferredSharedKeySize);
			out.writeInt(preferredSharedKeySize);
			log.debug(REMOTE_IP + "Sent Desired Prime Length of "
					+ preferredSharedKeySize);
			out.flush();
		} catch (final IOException e) {
			log.error(e);
		}
	}

	private boolean generatePrimeCandidate() throws Exception {
		BigInteger prime;
		for (int i = 0; i < genPrimes.size(); i++) {
			prime = (BigInteger) genPrimes.get(i);
			if (prime.equals(four)) {
				// do not transfer four
				pComposites.set(i, nMod4);
				qComposites.set(i, nMod4);
				// 3*3 mod 4 = 1
				nModPrimes.set(i, BigInteger.ONE);
				continue;
			}
			log.debug(REMOTE_IP + "Generating p and q mod " + prime);
			if (!sendSharingModPrime(prime)) {
				// should never get here, but other servers may handle things
				// differently
				return false;
			}
		}
		return true;
	}

	private void verifyN() throws Exception {
		log.debug(REMOTE_IP + "verifying candidate");
		BigInteger prime;
		for (int i = 0; i < verPrimes.size(); i++) {
			prime = (BigInteger) verPrimes.get(i);
			if (!sendSharingModPrime(prime)) {
				break;
			}
		}
	}

	/**
	 * Computes a sharing for a given prime
	 * 
	 * @param prime
	 *            the prime to use for the sharing
	 * @return true if the sharing is successful, false if the protocol needs to
	 *         be restarted.
	 * 
	 * @throws Exception
	 *             if the other party talks about a different prime
	 */
	private boolean sendSharingModPrime(final BigInteger prime)
			throws Exception {
		int position;

		// first retrieve the correct position of our prime lists
		if (super.actualState.equals(states[4])) {
			position = genPrimes.indexOf(prime);
		} else {
			position = verPrimes.indexOf(prime) + genPrimes.size();
		}

		// get the correct pre-computed Benaloh systems
		final BenalohEngine pubEngine = ((BenalohSystem) benalohSystems
				.get(prime)).getPubEngine();
		final BenalohEngine privEngine = ((BenalohSystem) benalohSystems
				.get(prime)).getPrivEngine();

		// set N mod p_i to 0, thus N is divided by p_i
		int nModPi = 0;
		BigInteger pModPrime;
		BigInteger qModPrime;
		BigInteger n;

		// and do the generation of N until it is no longer divided by p_i or
		// fail finally and must be newly generated.
		while (nModPi == 0) {
			if (super.actualState.equals(states[4])) {
				// p and q are not set, we can adopt as necessary
				log.debug(REMOTE_IP + "generating N mod " + prime);
				// get new random p and q
				BigInteger[] pq = generateNewPrimes(prime);
				pModPrime = pq[0];
				qModPrime = pq[1];
				// set them in the lists that are later used to reconstruct N
				// via CRT
				pComposites.set(position, pModPrime);
				qComposites.set(position, qModPrime);
			} else {
				// p and q are set, calculate values
				pModPrime = p.mod(prime);
				qModPrime = q.mod(prime);
			}

			// encrypt p mod p_i and q mod p_i
			final byte[] z1 = pubEngine.processBlock(pModPrime.toByteArray(),
					0, pModPrime.toByteArray().length);
			final byte[] z2 = pubEngine.processBlock(qModPrime.toByteArray(),
					0, qModPrime.toByteArray().length);

			// write them to the server
			out.write(GilboaProtocol.CONVERSION_MOD_PRIME);
			out.writeObject(prime);

			out.write(GilboaProtocol.REMOTE_Z1);
			out.writeInt(z1.length);
			out.write(z1);

			out.write(GilboaProtocol.REMOTE_Z2);
			out.writeInt(z2.length);
			out.write(z2);

			out.flush();

			n = BigInteger.ZERO;
			// make sure that both parties are at the same step of the protocol
			if (in.read() == GilboaProtocol.COMPUTED_N_) {
				final BigInteger remPrime = (BigInteger) in.readObject();

				// check that both parties operate on the same prime
				if (prime.equals(remPrime)) {
					// read in the servers computed result
					final byte[] z_data = new byte[in.readInt()];
					in.readFully(z_data);

					// and decrypt it
					n = new BigInteger(1, privEngine.processBlock(z_data, 0,
							z_data.length));

					// compute the final N mod p_i
					n = n.add((pModPrime.multiply(qModPrime)).mod(prime));
					n = n.mod(prime);

					// test whether N != 0
					if (n.equals(BigInteger.ZERO)) {
						// N is zero mod prime
						log.debug(REMOTE_IP + "Found that N mod " + prime
								+ " is 0.\n");
						if (super.actualState.equals(states[4])) {
							// generate new shares mod prime, we still can
							// adjust
							out.write(GilboaProtocol.N_MOD_PRIME_IS_0);
							out.flush();
						} else {
							// Testing rsaN failed.
							// Restart generation.
							log.info(REMOTE_IP + "Restarting generation. "
									+ "N is divisible by " + prime);
							out.write(GilboaProtocol.N_FAILED);

							// reset state machine to correct state
							super.actualState = states[3];

							// and restart generation of N
							out
									.write(GilboaProtocol.START_CANDIDATE_GENERATION);
							out.flush();

							// inform caller about failure
							return false;
						}
					} else {
						// successfully generated N mod prime
						nModPi = n.intValue();
						// set it in the list from which N is recomposed
						nModPrimes.set(position, n);
						// inform other party about success
						out.write(GilboaProtocol.COMPUTED_N_);
						out.flush();
					}
				} else {
					throw new Exception(PROTO_VIOL
							+ "Not talking about the same prime.");
				}
			}
			// finally set n to the actual value so that the process restarts if
			// it is 0
			nModPi = n.intValue();
		}
		// if N != 0 mod prime, return true
		return true;
	}

	private void sendBenalohSystems() throws Exception {
		final Enumeration en = benalohSystems.elements();
		BenalohSystem bs;
		while (en.hasMoreElements()) {
			bs = (BenalohSystem) en.nextElement();
			out.write(GilboaProtocol.BENALOH_SETUP);
			out.writeObject(bs.getPub().getY());
			out.writeObject(bs.getPub().getMessageBorder());
			out.writeObject(bs.getPub().getModulus());
			out.flush();
			if (in.read() == GilboaProtocol.BENALOH_SETUP) {
				log.debug(REMOTE_IP + "BenalohSystem for prime "
						+ bs.getPub().getMessageBorder() + " successful");
			} else {
				final String msg = PROTO_VIOL + "expecting BENALOH_SETUP";
				log.fatal(REMOTE_IP + msg);
				throw new Exception(msg);
			}
		}
	}

	protected void setFinalPQ() {
		p = PrimeUtils.chineseRemainder(pComposites, genPrimes);
		q = PrimeUtils.chineseRemainder(qComposites, genPrimes);
		log.debug(REMOTE_IP + "My generated p is " + p + " with "
				+ p.bitLength() + " bit.");
		log.debug(REMOTE_IP + "My generated q is " + q + " with "
				+ q.bitLength() + " bit.");
	}

}
