package net.sf.distrib_rsa;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import javax.net.ServerSocketFactory;
import javax.net.SocketFactory;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;
import javax.swing.JFrame;

import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohKeySerializer;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohPrivateKeyParameters;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohSystem;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternPrivateKeyParameters;
import net.sf.distrib_rsa.ui.AcceptRemoteCert;
import net.sf.distrib_rsa.ui.KeyGenerationParameters;
import net.sf.distrib_rsa.ui.LoadKeyStoreUI;
import net.sf.distrib_rsa.ui.NetSetupUI;
import net.sf.distrib_rsa.ui.SaveSettings;
import net.sf.distrib_rsa.ui.StandardDialogs;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;


/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class EnvironmentSetup {

	final static int TCP_SENDBUF_SIZE=32768;
	final static int TCP_RECVBUF_SIZE=4096;
	
	private static boolean listening = true;

	private static final Logger log = Logger.getLogger(EnvironmentSetup.class);

	private static final Properties prop = new Properties();

	public static final String PORT = "Port";

	public static final String KEY_STORE_FILE_NAME = "KeyStoreFileName";

	public static final String KEY_STORE_PASS = "KeyStorePass";

	public static final String KEY_PASS = "KeyPass";

	public static final String KEY_NAME = "KeyName";

	public static final String CERT_NAME = "CertificateName";

	public static final String KEY_SIZE = "KeySize";

	public static final String KEY_STORE_TYPE = "KeyStoreType";

	public static final String SERVER_MODE = "IsServer";

	public static final String SERVER_ADDRESS = "ServerAddress";

	public static final String PRIME_CERTAINTY = "PrimeCertainty";

	public static final String PROTOCOL = "Protocol";

//	public static final String[] PROTOCOLS = { "Gilboa '99", "Straub '04" };
	public static final String[] PROTOCOLS = { "Gilboa '99" };

	public static final String NCS_PRIV_KEY = "NaccacheSternPrivateKey";

	public static final String NCS_PUB_KEY = "NaccacheSternPublicKey";

	public static final String BENALOH_SYSTEM = "BenalohSystem";

	private static SSLContext sslContext;

	private static UnlockedKeyStore uks;

	private static KeyStore ks;

	private static SecureRandom random;

	private static NaccacheSternKeyParameters ncsPubKey;

	private static NaccacheSternPrivateKeyParameters ncsPrivKey;

	private static final int EXCHANGE_CERT = 1;

	private static final Hashtable benalohSystems = new Hashtable();

	private static int port;

	private static boolean isServer;

	private static String connectTo;

	private static String protocol;

	private static int keySize;

	private static int primeCertainty;

	/**
	 * Gets a SSLClientSocket (SSLSocket) from the SSLContext
	 * 
	 * @param s
	 *            The Socket that shall be converted to a SSL socket
	 * @return an SSLSocket for secure communication
	 * @throws UnknownHostException
	 * @throws IOException
	 */
	private static SSLSocket getSSLSocket(final Socket s)
			throws UnknownHostException, IOException {
		final SSLSocketFactory ssf = sslContext.getSocketFactory();
		final InetSocketAddress isa = (InetSocketAddress) s
				.getRemoteSocketAddress();
		final SSLSocket sock = (SSLSocket) ssf.createSocket(s, isa
				.getHostName(), isa.getPort(), true);
		return sock;
	}

	/**
	 * Initializes the SSLContext with the Certificates from the
	 * UnlockedKeyStore.
	 * 
	 * @throws NoSuchProviderException
	 * @throws KeyStoreException
	 * @throws IOException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws UnrecoverableKeyException
	 * @throws KeyManagementException
	 * @throws ClassNotFoundException
	 */
	private static void initSSLContext() throws KeyStoreException,
			NoSuchProviderException, NoSuchAlgorithmException,
			CertificateException, IOException, UnrecoverableKeyException,
			KeyManagementException, ClassNotFoundException {
		// generate a not password protected keyStore on the fly and copy
		// contents of uks to it
		if (!ks.containsAlias("default")) {
			final PrivateKey privKey = uks.getPrivateKey();
			try {
				ks.setKeyEntry("default", privKey, new char[0], uks
						.getCertificateChain());
			} catch (final Exception e) {
				log
						.fatal(
								"caught exception that privateKey is not really private",
								e);
				log.fatal("private key is: " + privKey);
			}
		}
		final Enumeration en = uks.ks.aliases();
		int i = 0;
		while (en.hasMoreElements()) {
			final String customName = en.nextElement().toString();
			final Certificate cert = uks.ks.getCertificate(customName);
			if (ks.getCertificate(customName) == null) {
				ks.setCertificateEntry(customName, cert);
			}
			i++;
		}

		final SecureRandom random = EnvironmentSetup.getSecureRandom();

		final KeyManagerFactory kmf = KeyManagerFactory
				.getInstance(KeyManagerFactory.getDefaultAlgorithm());

		kmf.init(ks, new char[0]);
		final TrustManagerFactory tmf = TrustManagerFactory
				.getInstance(TrustManagerFactory.getDefaultAlgorithm());
		tmf.init(ks);
		sslContext = SSLContext.getInstance("SSLv3");
		sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), random);

	}

	/**
	 * initLog4J() initializes the Logger to reasonable defaults.
	 * 
	 */
	private static void initLog4J() {
		final Properties log4jproperties = new Properties();
		// configure two loggers: stdout (console) and R (file) logging
		log4jproperties.setProperty("log4j.rootLogger", "DEBUG, C, R");

		// configure A1 for console
		log4jproperties.setProperty("log4j.appender.C",
				"org.apache.log4j.ConsoleAppender");
		// configure custom Layout for console
		log4jproperties.setProperty("log4j.appender.C.layout",
				"org.apache.log4j.PatternLayout");
		log4jproperties.setProperty(
				"log4j.appender.C.layout.ConversionPattern",
				"[%-5p] [%d{HH:mm:ss}] %c.%M:%L %m\n");
		log4jproperties.setProperty("log4j.appender.C.threshold", "INFO");

		// configure file for file appending
		log4jproperties.setProperty("log4j.appender.R",
				"org.apache.log4j.RollingFileAppender");
		// set filename
		String hostname = null;
		System.out.println("Running on Java "
				+ System.getProperty("java.version"));
		if (!System.getProperty("java.version").startsWith("1.4")) {
			hostname = System.getenv("HOSTNAME");
		}
		if ((hostname == null) || "".equals(hostname)) {
			hostname = "localhost";
		}
		final SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
		log4jproperties.setProperty("log4j.appender.R.File", hostname + "_"
				+ df.format(new Date()) + ".log");
		// set max. filesize
		log4jproperties.setProperty("log4j.appender.R.MaxFileSize", "300KB");
		// set no. of backups
		log4jproperties.setProperty("log4j.appender.R.MaxBackupIndex", "3");
		// set custom logging layout
		log4jproperties.setProperty("log4j.appender.R.layout",
				"org.apache.log4j.PatternLayout");
		log4jproperties.setProperty(
				"log4j.appender.R.layout.ConversionPattern",
				"[%-5p] [%d{yyyy-MM-dd HH:mm:ss}] %m\n");
		// set custom logging level
		log4jproperties.setProperty("log4j.appender.R.threshold", "DEBUG");
		log4jproperties.setProperty(
				"log4j.logger.de.uni_bremen.informatik.lippold", "DEBUG");

		PropertyConfigurator.configure(log4jproperties);
	}

	public static void main(final String[] args) throws KeyManagementException,
			KeyStoreException, NoSuchAlgorithmException, CertificateException,
			FileNotFoundException, UnrecoverableKeyException, IOException,
			NoSuchProviderException, ClassNotFoundException {

		// Initialize Security Provider with BC
		Security.addProvider(new BouncyCastleProvider());

		// Initialize logging
		initLog4J();

		ks = KeyStore.getInstance("PKCS12", "BC");
		ks.load(null, new char[0]);

		if (args.length == 1) {
			log.debug("Loading settings from " + args[0]);

			// read in pre-configured properties for running without user
			// interface
			try {
				final FileInputStream is = new FileInputStream(args[0]);
				prop.load(is);
			} catch (final Exception e) {
				log.fatal("Could not load PropertyFile " + args[0]
						+ " with default parameters.\nReason:\n" + e.toString()
						+ "\n If you want to generate a new property file,"
						+ " please do not supply any arguments.");
				log.fatal("Exception was: ", e);
				System.exit(1);
			}
		} else {
			StandardDialogs.defaultFrame = new JFrame();
			log.debug("Questioning user about parameters");
			// Question user about parameters and save them
			final UnlockedKeyStore uks = LoadKeyStoreUI.getUserKeyStore();
			uks.addToProperties(prop);
			uks.save();
			KeyGenerationParameters.getKeySize(prop);
			keySize = Integer.parseInt(prop.getProperty(KEY_SIZE, "1024"));
			KeyGenerationParameters.getPrimeCertainty(prop);
			primeCertainty = Integer
					.parseInt(prop.getProperty(PRIME_CERTAINTY));
			NetSetupUI.getOperationMode(prop);
			if (Boolean.valueOf(prop.getProperty(SERVER_MODE)).booleanValue() == false) {
				NetSetupUI.getPreferredProtocol(prop);
				protocol = prop.getProperty(PROTOCOL);
				// KeyGenerationParameters.generateNCSKeyPair(prop);
				KeyGenerationParameters.generateBenalohSystems(prop);
			}
		}
		readSettings();

		// Ask user to save working new properties
		if (args.length == 0) {
			SaveSettings.save(prop);
		}

		// Save a lot of memory :)
		prop.clear();
		System.gc();
		Thread.yield();

		if (isServer) {
			final ServerSocketFactory ssf = ServerSocketFactory.getDefault();
			final ServerSocket listenSocket = ssf.createServerSocket(port);
			listenSocket.setReceiveBufferSize(TCP_RECVBUF_SIZE);

			log.info("Server ready and listening");
			while (listening) {
				try {
					final Socket sock = listenSocket.accept();
					log.debug("Send buffer size is " + sock.getSendBufferSize());
					sock.setSendBufferSize(TCP_SENDBUF_SIZE);
					setupSSLConnection(sock, true);
				} catch (final Exception e) {
					log.fatal("Exception during connection preparation:", e);
				}
			}
		} else {
			final SocketFactory sf = SocketFactory.getDefault();
			log.info("connecting as client to " + connectTo + ":" + port);
			final Socket clientSocket = sf.createSocket(connectTo, port);
			clientSocket.setReceiveBufferSize(TCP_RECVBUF_SIZE);
			log.debug("Send buffer size is " + clientSocket.getSendBufferSize());
			clientSocket.setSendBufferSize(TCP_SENDBUF_SIZE);
			setupSSLConnection(clientSocket, false);
		}
		StandardDialogs.disposeUI();
	}

	/**
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws KeyStoreException
	 * @throws ClassNotFoundException
	 * 
	 */
	private static void readSettings() throws KeyStoreException,
			NoSuchAlgorithmException, CertificateException,
			FileNotFoundException, IOException, ClassNotFoundException {
		final String keyStoreFileName = prop.getProperty(KEY_STORE_FILE_NAME,
				"keyStore.ks");
		final String keyStoreType = prop.getProperty(KEY_STORE_TYPE, "PKCS12");
		final String keyStorePass = prop.getProperty(KEY_STORE_PASS);
		final String certName = prop.getProperty(CERT_NAME);
		final String keyName = prop.getProperty(KEY_NAME);
		final String keyPass = prop.getProperty(KEY_PASS);
		if ((keyStorePass == null) || (keyPass == null) || (keyName == null)
				|| (certName == null)) {
			log.error("You didn't supply all necessary parameters\nExiting");
			System.exit(1);
		}
		uks = new UnlockedKeyStore(keyStoreFileName, keyStoreType,
				keyStorePass, certName, keyName, keyPass);

		keySize = Integer.parseInt(prop.getProperty(KEY_SIZE, "1024"));
		if (keySize < 1) {
			log.error("The size of the key to be generated is <1.\nExiting");
			System.exit(1);
		}

		/** Don't need NCS key pair atm. */
		// final String pubStr = prop.getProperty(NCS_PUB_KEY);
		// ncsPubKey = NaccacheSternKeySerializationFactory.deserialize(Base64
		// .decode(pubStr));
		// final String privStr = prop.getProperty(NCS_PRIV_KEY);
		// ncsPrivKey = (NaccacheSternPrivateKeyParameters)
		// NaccacheSternKeySerializationFactory
		// .deserialize(Base64.decode(privStr));
		port = Integer.parseInt(prop.getProperty(PORT, "4444"));
		isServer = Boolean.valueOf(prop.getProperty(SERVER_MODE))
				.booleanValue();
		if (!isServer) {
			loadBenalohSystems();
			connectTo = prop.getProperty(SERVER_ADDRESS);
		}
		protocol = prop.getProperty(PROTOCOL);
		primeCertainty = Integer.parseInt(prop.getProperty(PRIME_CERTAINTY));
	}

	private static void setupSSLConnection(final Socket sock, boolean isServer)
			throws IOException, KeyManagementException, KeyStoreException,
			NoSuchProviderException, NoSuchAlgorithmException,
			CertificateException, UnrecoverableKeyException,
			ClassNotFoundException {

		// Exchange certificates, check that they exist or add them to uks if
		// user confirms
		final BufferedOutputStream out = new BufferedOutputStream(sock
				.getOutputStream());
		final BufferedInputStream bis = new BufferedInputStream(sock
				.getInputStream());
		X509Certificate remoteCert = null;
		try {
			sock.setTrafficClass(0x18);
			sock.setTcpNoDelay(true);
			sock.setReceiveBufferSize(TCP_RECVBUF_SIZE);
			sock.setSendBufferSize(TCP_SENDBUF_SIZE);
		} catch (SocketException e1) {
			log.info("could not set desired traffic class");
		}
		if (isServer) {
			out.write(EXCHANGE_CERT);
			final ObjectOutputStream oos = new ObjectOutputStream(out);
			oos.writeObject(getUnlockedKeyStore().getCertificate());
			oos.flush();
			out.flush();
			if (bis.read() == EXCHANGE_CERT) {
				final ObjectInputStream ois = new ObjectInputStream(bis);
				remoteCert = (X509Certificate) ois.readObject();
			} else {
				log.fatal("Certificate exchange failed. Returning.");
				return;
			}
		} else {
			if (bis.read() == EXCHANGE_CERT) {
				final ObjectInputStream ois = new ObjectInputStream(bis);
				remoteCert = (X509Certificate) ois.readObject();
			} else {
				log.fatal("Certificate exchange failed. Returning.");
				return;
			}
			out.write(EXCHANGE_CERT);
			final ObjectOutputStream oos = new ObjectOutputStream(out);
			oos.writeObject(getUnlockedKeyStore().getCertificate());
			oos.flush();
			out.flush();
		}
		if (remoteCert != null) {
			final UnlockedKeyStore uks = getUnlockedKeyStore();
			if (!isKnownCertificate(remoteCert)) {
				if (AcceptRemoteCert.acceptCert(remoteCert)) {
					uks
							.addCert(remoteCert.getSubjectDN().getName(),
									remoteCert);
					ks.setCertificateEntry(remoteCert.getSubjectDN().getName(),
							remoteCert);
				}
			}
			remoteCert.checkValidity();
		}

		// initialize SSL Connection
		initSSLContext();
		final SSLSocket sslSock = getSSLSocket(sock);
		try {
			sslSock.setTrafficClass(0x18);
			sslSock.setTcpNoDelay(true);
		} catch (SocketException e1) {
			log.info("could not set desired traffic class");
		}
		if (isServer) {
			sslSock.setUseClientMode(false);
		} else {
			sslSock.setUseClientMode(true);
		}
		final SSLHandshakeListener sslHL = new SSLHandshakeListener(isServer);
		if (!isServer) {
			sslHL.setProtocol(getProtocol());
		}
		sslSock.addHandshakeCompletedListener(sslHL);
		if (isServer) {
			sslSock.setNeedClientAuth(true);
		}
		sslSock.setEnableSessionCreation(true);
		sslSock.startHandshake();
	}

	public static String getProtocol() {
		return protocol;
	}

	public static UnlockedKeyStore getUnlockedKeyStore() {
		return uks;
	}

	public void stopServer() {
		listening = false;
	}

	public static SecureRandom getSecureRandom()
			throws NoSuchAlgorithmException, NoSuchProviderException {

		if (random == null) {
			try {
				random = SecureRandom.getInstance("SHA1PRNG", "SUN");
			} catch (final Exception e) {
				random = new SecureRandom();
			}

			if (new File("/dev/urandom").exists()) {
				final byte[] salt = new byte[8192];
				try {
					new FileInputStream("/dev/urandom").read(salt);
					random.setSeed(salt);
				} catch (final Exception e) {
					log.info("/dev/urandom not available");
				}
			}
		}

		return random;
	}

	public static int getDesiredKeySize() {
		return keySize;
	}

	private static boolean isKnownCertificate(final Certificate cert) {
		if (uks.containsCertificate(cert)) {
			return true;
		}
		return false;
	}

	/**
	 * @return Returns the primeCertainty.
	 */
	public static int getPrimeCertainty() {
		return primeCertainty;
	}

	public static NaccacheSternPrivateKeyParameters getNcsPrivKey() {
		return ncsPrivKey;
	}

	public static NaccacheSternKeyParameters getNcsPubKey() {
		return ncsPubKey;
	}

	public static Hashtable getBenalohSystems() {
		return benalohSystems;
	}

	private static void loadBenalohSystems() throws IOException,
			ClassNotFoundException {
		final Enumeration en = prop.keys();
		while (en.hasMoreElements()) {
			final String key = (String) en.nextElement();
			if (key.startsWith(BENALOH_SYSTEM)) {
				final String privEnc = prop.getProperty(key);
				final BenalohPrivateKeyParameters priv = (BenalohPrivateKeyParameters) BenalohKeySerializer
						.deserialize(Base64.decode(privEnc));
				final BenalohSystem bs = new BenalohSystem(priv);
				benalohSystems.put(bs.getPrime(), bs);
			}
		}
	}
}
