package net.sf.distrib_rsa;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Date;

import javax.net.ssl.HandshakeCompletedEvent;
import javax.net.ssl.HandshakeCompletedListener;
import javax.net.ssl.SSLSocket;

import net.sf.distrib_rsa.protocols.RSAParameters;
import net.sf.distrib_rsa.protocols.computeD.ComputeKey;
import net.sf.distrib_rsa.protocols.computeD.ComputeKeyClient;
import net.sf.distrib_rsa.protocols.computeD.ComputeKeyServer;
import net.sf.distrib_rsa.protocols.gilboaSharing.GilboaClient;
import net.sf.distrib_rsa.protocols.gilboaSharing.GilboaServer;
import net.sf.distrib_rsa.protocols.straubSharing.StraubClient;
import net.sf.distrib_rsa.protocols.straubSharing.StraubServer;
import net.sf.distrib_rsa.ui.StandardDialogs;

import org.apache.log4j.Logger;


/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class SSLHandshakeListener implements HandshakeCompletedListener {

	/**
	 * @uml.property  name="isServer"
	 */
	private final boolean isServer;

	private static final Logger log = Logger
			.getLogger(SSLHandshakeListener.class);

	/**
	 * @uml.property  name="thread"
	 */
	private Thread thread = null;

	/**
	 * @uml.property  name="protocol"
	 */
	private String protocol = "";

	/**
	 * @uml.property  name="in"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.lang.String"
	 */
	private ObjectInputStream in;

	/**
	 * @uml.property  name="out"
	 */
	private ObjectOutputStream out;

	public SSLHandshakeListener(final boolean isServer) {
		this.isServer = isServer;
	}

	/**
	 * The HandshakeCompleted
	 * 
	 * @see javax.net.ssl.HandshakeCompletedListener#handshakeCompleted(javax.net.ssl.HandshakeCompletedEvent)
	 */
	public void handshakeCompleted(final HandshakeCompletedEvent e) {
		final SSLSocket s = e.getSocket();
		InputStreamStatistics iss = null;
		//OutputStreamStatistics oss = null;
		try {
			//oss = new OutputStreamStatistics(s.getOutputStream());
			out = new ObjectOutputStream(s.getOutputStream());
			final String REMOTE_IP = s.getInetAddress() + " ";
			final String REMOTE_ID = "IP: " + REMOTE_IP + "\nCertificate: "
					+ s.getSession().getPeerCertificates()[0];
			log.info("Starting distributed RSA key generation with client "
					+ REMOTE_ID);
			iss = new InputStreamStatistics(s.getInputStream());
			in = new ObjectInputStream(iss);

			if (isServer) {
				log.info("Getting desired Protocol");
				protocol = (String) in.readObject();
				log.info("Protocol is " + protocol);
				if (protocol.equals(EnvironmentSetup.PROTOCOLS[0])) {
					thread = new GilboaServer(REMOTE_IP, in, out);
				} else if (protocol.equals(EnvironmentSetup.PROTOCOLS[1])) {
					thread = new StraubServer(s);
				}
			} else {
				protocol = EnvironmentSetup.getProtocol();
				log.info("Using protocol " + protocol);
				out.writeObject(protocol);
				out.flush();
				if (protocol.equals(EnvironmentSetup.PROTOCOLS[0])) {
					thread = new GilboaClient(REMOTE_IP, in, out);
				} else if (protocol.equals(EnvironmentSetup.PROTOCOLS[1])) {
					thread = new StraubClient(
							s);
				}
			}
			synchronized (thread) {
				log.info("Starting sieving");
				thread.start();
				try {
					thread.wait();
				} catch (final InterruptedException ex) {
				}
			}
			log.info("Sieving finished, now constructing shared RSA key");
			// now, p,q and the rsaN are generated and both parties trust it.
			// construct a RSA Key from it
			final RSAParameters param = (RSAParameters) thread;
			if (param.wasSuccessful()) {
				final BigInteger p = param.getP();
				final BigInteger q = param.getQ();
				final BigInteger rsaN = param.getRsaN();
				final SecureRandom rand = EnvironmentSetup.getSecureRandom();
				final int certainty = EnvironmentSetup.getPrimeCertainty();
				if (isServer) {
					thread = new ComputeKeyServer(p, q, rsaN, in, out,
							REMOTE_IP, certainty, rand);
				} else {
					thread = new ComputeKeyClient(p, q, rsaN, in, out,
							REMOTE_IP, certainty, rand);
				}
				synchronized (thread) {
					log.info("Starting key generation");
					thread.start();
					try {
						thread.wait();
					} catch (final InterruptedException ex) {
					}
				}
				log.info("Finished distributed RSA key generation with client "
						+ REMOTE_ID);
				s.close();
				final ComputeKey ck = (ComputeKey) thread;
				if (ck.wasSuccessful()) {
					final PKCS8EncodedKeySpec privKey = ck.getPrivKey();
					final X509EncodedKeySpec pubKey = ck.getPubKey();
					if ((privKey != null) && (pubKey != null)) {
						final String ip = s.getInetAddress().getHostAddress();
						// this should be writeable ;)
						final String userDir = System.getProperty("user.home");
						final Date d = new Date();
						String prefix;
						if (isServer) {
							prefix = "Server_" + ip + "_" + d.getTime();
						} else {
							prefix = "Client_" + ip + "_" + d.getTime();
						}
						final File privKeyFile = new File(userDir
								+ File.separator + prefix + "PrivateKey"
								+ ".pkcs8");
						final File pubKeyFile = new File(userDir
								+ File.separator + prefix + "PublicKey"
								+ ".x509");
						try {
							log.info("=============");
							log.info("Saving private Key to "
									+ privKeyFile.getAbsolutePath());
							FileOutputStream fos = new FileOutputStream(
									privKeyFile);
							fos.write(privKey.getEncoded());
							fos.flush();
							fos.close();
							log
									.info("If you want to use your new keys with openssl, try");
							log.info("openssl pkcs8 -inform DER -nocrypt -in "
									+ privKeyFile.getAbsolutePath());
							log
									.info("You can reconstruct the public key from the pem encoded openssl output by using");
							log
									.info("openssl rsa -inform pem -in key.pem -pubout");
							log.info("All together, output to key.pem:");
							log
									.info("openssl pkcs8 -inform DER -nocrypt -in "
											+ privKeyFile.getAbsolutePath()
											+ " -out key.pem && openssl rsa -inform pem -in key.pem -pubout >> key.pem");
							log
									.warn("The PRIVATE keys created with this software are _not_ openssl compatible");
							log
									.info("because openssl uses the Chinese Remainder Theorem for decryption");
							log
									.info("Since neither party knows p nor q, we cannot give CRT values");
							log.info("=============");
							log.info("");
							log.info("Saving public Key to "
									+ pubKeyFile.getAbsolutePath());
							fos = new FileOutputStream(pubKeyFile);
							fos.write(pubKey.getEncoded());
							fos.flush();
							fos.close();
						} catch (final IOException ex) {
							log.fatal(ex);
							log
									.fatal("Probably saving keys was not successful");
							log.info("Your key consists of d, e and rsaN.");
							log.info("Please check the logs for details");
						}
					}
				} else {
					log.fatal("Key generation failed");
					log.info("Check logs for details");
				}
			} else {
				log.fatal("Prime generation failed");
				log.info("Check logs for details");
			}
		} catch (final Exception ex) {
			log.fatal("Error in SSL Handshake Listener:", ex);
		}
		
		long inBytes = iss.getByteCount();
		//long outBytes = oss.getByteCount();
		log.info(inBytes + " bytes were received");
		//log.info(outBytes + " bytes were sent");
		//log.info((inBytes + outBytes) + " bytes total");
		final long seconds = iss.getRunningTime();
		final long minutes = seconds / 60;
		final long hours = seconds / 3600;
		log.info("Generation took " + hours + "h " + minutes % 60 + "m " + seconds % 60 + "s");
		log.info("On average, " + iss.getBytesPerSecond() + " bytes/second were received.");
		//log.info("On average, " + oss.getBytesPerSecond() + " bytes/second were sent.");
		//log.info("Average traffic was " + (iss.getBytesPerSecond() + oss.getBytesPerSecond()) + " bytes/second");
		StandardDialogs.disposeUI();
	}

	/**
	 * @param protocol  the protocol to set
	 * @uml.property  name="protocol"
	 */
	public void setProtocol(final String s) {
		protocol = s;
	}

}
