package net.sf.distrib_rsa.cryptosystems.benaloh;

import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;

import net.sf.distrib_rsa.EnvironmentSetup;

import org.apache.log4j.Logger;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class BenalohSystem {

	private final static Logger log = Logger.getLogger(BenalohSystem.class);

	/**
	 * @uml.property  name="debug"
	 */
	boolean debug = true;

	/**
	 * @uml.property  name="pub"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	private final BenalohKeyParameters pub;

	/**
	 * @uml.property  name="priv"
	 * @uml.associationEnd  
	 */
	private final BenalohPrivateKeyParameters priv;

	/**
	 * @uml.property  name="pubEngine"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	private final BenalohEngine pubEngine = new BenalohEngine();

	/**
	 * @uml.property  name="privEngine"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	private final BenalohEngine privEngine = new BenalohEngine();

	public BenalohSystem(final int strength, final BigInteger prime)
			throws NoSuchAlgorithmException, NoSuchProviderException {
		final BenalohKeyGenerationParameters bkgp = new BenalohKeyGenerationParameters(
				EnvironmentSetup.getSecureRandom(), strength, prime,
				EnvironmentSetup.getPrimeCertainty());
		final BenalohKeyPairGenerator bkpg = new BenalohKeyPairGenerator();
		bkpg.init(bkgp);
		final AsymmetricCipherKeyPair ackp = bkpg.generateKeyPair();
		log.debug("Initializing pub engine");
		pubEngine.setDebug(debug);
		pub = (BenalohKeyParameters) ackp.getPublic();
		pubEngine.init(true, pub);
		log.debug("Initializing priv engine");
		privEngine.setDebug(debug);
		priv = (BenalohPrivateKeyParameters) ackp.getPrivate();
		priv.setupLookupList();
		privEngine.init(false, priv);
	}

	public BenalohSystem(final BenalohKeyParameters pub) {
		this.pub = pub;
		pubEngine.init(true, pub);
		priv = null;
	}

	public BenalohSystem(final BenalohPrivateKeyParameters priv) {
		this.priv = priv;
		pub = new BenalohKeyParameters(priv.getY(), priv
				.getMessageBorder(), priv.getModulus());
		privEngine.init(false, priv);
		pubEngine.init(true, pub);
	}

	/**
	 * @return the prime
	 */
	public final BigInteger getPrime() {
		return pub.getMessageBorder();
	}

	/**
	 * @return  the priv
	 * @uml.property  name="priv"
	 */
	public final BenalohPrivateKeyParameters getPriv() {
		return priv;
	}

	/**
	 * @return  the privEngine
	 * @uml.property  name="privEngine"
	 */
	public final BenalohEngine getPrivEngine() {
		return privEngine;
	}

	/**
	 * @return  the pub
	 * @uml.property  name="pub"
	 */
	public final BenalohKeyParameters getPub() {
		return pub;
	}

	/**
	 * @return  the pubEngine
	 * @uml.property  name="pubEngine"
	 */
	public final BenalohEngine getPubEngine() {
		return pubEngine;
	}

}
