package net.sf.distrib_rsa.cryptosystems.naccacheStern;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Vector;

import net.sf.distrib_rsa.cryptosystems.PrimeUtils;

import org.bouncycastle.crypto.AsymmetricBlockCipher;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.InvalidCipherTextException;

/**
 * NaccacheStern Engine. For details on this cipher, please see
 * http://www.gemplus.com/smart/rd/publications/pdf/NS98pkcs.pdf
 */
public class NaccacheSternEngine implements AsymmetricBlockCipher {
	/**
	 * @uml.property name="forEncryption"
	 */
	private boolean forEncryption;

	/**
	 * @uml.property name="key"
	 * @uml.associationEnd
	 */
	private NaccacheSternKeyParameters key;

	/**
	 * @uml.property name="debug"
	 */
	private boolean debug = false;

	/**
	 * @uml.property name="threads"
	 * @uml.associationEnd multiplicity="(0 -1)"
	 *                     inverse="this$0:de.uni_bremen.informatik.lippold.cryptosystems.naccacheStern.NaccacheSternEngine$DecryptBySmallPrime"
	 */
	private Vector threads;

	/**
	 * @uml.property name="waitFor"
	 */
	private final Object waitFor = new Object();

	/**
	 * @uml.property name="plain"
	 * @uml.associationEnd multiplicity="(0 -1)"
	 *                     elementType="java.math.BigInteger"
	 */
	private Vector plain;

	/**
	 * @uml.property name="certificate"
	 */
	private BigInteger certificate;

	/**
	 * @uml.property name="processorCount"
	 */
	private final int processorCount;

	public NaccacheSternEngine() {
		processorCount = 1;
	}

	public NaccacheSternEngine(final int processorCnt) {
		processorCount = processorCnt;
	}

	/**
	 * Initializes this algorithm. Must be called before all other Functions.
	 * 
	 * @see org.bouncycastle.crypto.AsymmetricBlockCipher#init(boolean,
	 *      org.bouncycastle.crypto.CipherParameters)
	 */
	public void init(final boolean forEncryption, final CipherParameters param) {
		this.forEncryption = forEncryption;

		// construct lookup table for faster decryption if necessary
		if (this.forEncryption) {
			key = (NaccacheSternKeyParameters) param;
		} else {
			final NaccacheSternPrivateKeyParameters privKey = (NaccacheSternPrivateKeyParameters) param;
			key = privKey;
		}
	}

	/**
	 * @param debug
	 *            the debug to set
	 * @uml.property name="debug"
	 */
	public void setDebug(final boolean debug) {
		this.debug = debug;
	}

	/**
	 * Returns the input block size of this algorithm.
	 * 
	 * @see org.bouncycastle.crypto.AsymmetricBlockCipher#getInputBlockSize()
	 */
	public int getInputBlockSize() {
		if (forEncryption) {
			// We can only encrypt values up to lowerSigmaBound
			return (key.getSigma().bitLength() + 7) / 8 - 1;
		} else {
			// We pad to modulus-size bytes for easier decryption.
			return key.getModulus().toByteArray().length;
		}
	}

	/**
	 * Returns the output block size of this algorithm.
	 * 
	 * @see org.bouncycastle.crypto.AsymmetricBlockCipher#getOutputBlockSize()
	 */
	public int getOutputBlockSize() {
		if (forEncryption) {
			// encrypted Data is always padded up to modulus size
			return key.getModulus().toByteArray().length;
		} else {
			// decrypted Data has upper limit lowerSigmaBound
			return (key.getSigma().bitLength() + 7 / 8) + 1;
		}
	}

	/**
	 * Process a single Block using the Naccache-Stern algorithm.
	 * 
	 * @see org.bouncycastle.crypto.AsymmetricBlockCipher#processBlock(byte[],
	 *      int, int)
	 */
	public byte[] processBlock(final byte[] input, final int inOff,
			final int len) throws InvalidCipherTextException {

		if (!forEncryption && (len < getInputBlockSize())) {
			// At decryption make sure that we receive padded data blocks
			throw new InvalidCipherTextException(
					"BlockLength does not match modulus for Naccache-Stern cipher.\n");
		}

		byte[] block;

		if ((inOff != 0) || (len != input.length)) {
			block = new byte[len];
			System.arraycopy(input, inOff, block, 0, len);
		} else {
			block = input;
		}

		// transform input into BigInteger
		final BigInteger input_converted = new BigInteger(1, block);

		if (forEncryption && (input_converted.compareTo(key.getSigma()) > 0)) {
			throw new DataLengthException(
					"input too large for Naccache-Stern cipher.\n");
		}
		if (debug) {
			System.out.println("input as BigInteger: " + input_converted);
		}
		byte[] output;
		if (forEncryption) {
			output = encrypt(input_converted);
		} else {
			output = decrypt(input_converted);
		}

		return output;
	}

	/**
	 * Encrypts a BigInteger aka Plaintext with the public key. Uses
	 * probabilistic encryption if a certificate is set.
	 * 
	 * @param plain
	 *            The BigInteger to encrypt
	 * @return The byte[] representation of the encrypted BigInteger (i.e.
	 *         crypted.toByteArray())
	 */
	public byte[] encrypt(final BigInteger plain) {
		// Always return modulus size values 0-padded at the beginning
		// 0-padding at the beginning is correctly parsed by BigInteger :)
		final byte[] output = key.getModulus().toByteArray();
		Arrays.fill(output, Byte.parseByte("0"));
		BigInteger encrypted = key.getY().modPow(plain, key.getModulus());

		// Certificate is the uniformly at random generated obfuscating element
		// from Z_n
		if (certificate != null) {
			encrypted = certificate.modPow(key.getSigma(), key.getModulus())
					.multiply(encrypted).mod(key.getModulus());
		}

		final byte[] tmp = encrypted.toByteArray();
		System
				.arraycopy(tmp, 0, output, output.length - tmp.length,
						tmp.length);
		if (debug) {
			System.out.println("Encrypted value is:  "
					+ new BigInteger(1, output));
		}
		return output;
	}

	/**
	 * Adds the contents of two encrypted blocks mod sigma
	 * 
	 * @param block1
	 *            the first encrypted block
	 * @param block2
	 *            the second encrypted block
	 * @return an encrypted block, so that decrypt(retval) = ( decrypt(block1) +
	 *         decrypt(block2) ) % sigma
	 * @throws InvalidCipherTextException
	 */
	public byte[] addCryptedBlocks(final byte[] block1, final byte[] block2)
			throws InvalidCipherTextException {
		final BigInteger m1Crypt = new BigInteger(1, block1);
		final BigInteger m2Crypt = new BigInteger(1, block2);

		// check for correct blocksize
		if ((m1Crypt.compareTo(key.getModulus()) >= 0)
				|| (m2Crypt.compareTo(key.getModulus()) >= 0)) {
			throw new InvalidCipherTextException(
					"BlockLength too large for addition");
		}

		// calculate resulting block
		BigInteger m1m2Crypt = m1Crypt.multiply(m2Crypt);
		m1m2Crypt = m1m2Crypt.mod(key.getModulus());
		if (debug) {
			System.out.println("c(m1) as BigInteger:......... " + m1Crypt);
			System.out.println("c(m2) as BigInteger:......... " + m2Crypt);
			System.out.println("(c(m1)*c(m2))%n = c(m1+m2)%n: " + m1m2Crypt);
		}

		final byte[] output = key.getModulus().toByteArray();
		Arrays.fill(output, Byte.parseByte("0"));
		System.arraycopy(m1m2Crypt.toByteArray(), 0, output, output.length
				- m1m2Crypt.toByteArray().length,
				m1m2Crypt.toByteArray().length);

		return output;
	}

	/**
	 * Multiplies block1 by value (mod sigma).
	 * 
	 * @param block1
	 *            The encrypted block to be multiplied.
	 * @param value
	 *            The value by which it shall be multiplied
	 * @return an encrypted block, so that decrypt(retval) = ( decrypt(block1) *
	 *         value ) % sigma
	 * @throws InvalidCipherTextException
	 */
	public byte[] multiplyCryptedBlock(final byte[] block1,
			final BigInteger value) throws InvalidCipherTextException {

		final BigInteger m1Crypt = new BigInteger(1, block1);
		if (m1Crypt.compareTo(key.getModulus()) >= 0) {
			throw new InvalidCipherTextException(
					"BlockLength too large for multiplication.\n");
		}

		// calculate resulting block
		final BigInteger m1m2Crypt = m1Crypt.modPow(value, key.getModulus());
		if (debug) {
			System.out.println("c(m1) as BigInteger:....... " + m1Crypt);
			System.out.println("m2 as BigInteger:.......... " + value);
			System.out.println("(c(m1)^m2)%n = c(m1*m2)%n: " + m1m2Crypt);
		}

		final byte[] output = key.getModulus().toByteArray();
		Arrays.fill(output, Byte.parseByte("0"));
		System.arraycopy(m1m2Crypt.toByteArray(), 0, output, output.length
				- m1m2Crypt.toByteArray().length,
				m1m2Crypt.toByteArray().length);

		return output;
	}

	/**
	 * Convenience Method for data exchange with the cipher.
	 * 
	 * Determines blocksize and splits data to blocksize.
	 * 
	 * @param data
	 *            the data to be processed
	 * @return the data after it went through the NaccacheSternEngine.
	 * @throws InvalidCipherTextException
	 */
	public byte[] processData(final byte[] data)
			throws InvalidCipherTextException {
		if (debug) {
			System.out.println();
		}
		final BigInteger dataConverted = new BigInteger(1, data);
		if (dataConverted.compareTo(key.getSigma()) > 0) {
			final int inBlocksize = getInputBlockSize();
			final int outBlocksize = getOutputBlockSize();
			if (debug) {
				System.out.println("Input blocksize is:  " + inBlocksize
						+ " bytes");
				System.out.println("Output blocksize is: " + outBlocksize
						+ " bytes");
				System.out.println("Data has length:.... " + data.length
						+ " bytes");
			}
			int datapos = 0;
			int retpos = 0;
			final byte[] retval = new byte[(data.length / inBlocksize + 1)
					* outBlocksize];
			while (datapos < data.length) {
				byte[] tmp;
				if (datapos + inBlocksize < data.length) {
					tmp = processBlock(data, datapos, inBlocksize);
					datapos += inBlocksize;
				} else {
					tmp = processBlock(data, datapos, data.length - datapos);
					datapos += data.length - datapos;
				}
				if (debug) {
					System.out.println("new datapos is " + datapos);
				}
				if (tmp != null) {
					for (int i = 0; i < tmp.length; i++) {
						retval[i + retpos] = tmp[i];
					}
					retpos += tmp.length;
				} else {
					if (debug) {
						System.out.println("cipher returned null");
					}
					throw new InvalidCipherTextException("cipher returned null");
				}
			}
			final byte[] ret = new byte[retpos];
			System.arraycopy(retval, 0, ret, 0, retpos);
			if (debug) {
				System.out.println("returning " + ret.length + " bytes");
			}
			return ret;
		} else {
			if (debug) {
				System.out
						.println("data size is less then input block size, processing directly");
			}
			return processBlock(data, 0, data.length);
		}
	}

	/**
	 * Set the certificate for this engine. This is a BigInteger that is used by
	 * probabilistic encryption to conceal the actual encrypted value. This also
	 * allows for changing certificates before an encryption process. The
	 * certificate remains valid for this enigne until a new one is set.
	 * 
	 * @param certificate
	 *            The BigInteger used as a certificate to conceal the encrypted
	 *            value.
	 * @uml.property name="certificate"
	 */
	public void setCertificate(final BigInteger certificate) {
		this.certificate = certificate;
	}

	/**
	 * Returns the value of the decryption modulo the primes in the private key.
	 * This is useful for using NaccacheStern as a multiple Benaloh system in
	 * which one can calculate modulo every prime in the private key.
	 * 
	 * @return Two Vectors, the first with the decryption values, the second
	 *         with the primes. The decryption values are in the order that the
	 *         primes have. Thus, the return value can be feeded directly into
	 *         CRT.
	 * 
	 */
	public Vector[] getDecryptionModuloPrimes() {
		final NaccacheSternPrivateKeyParameters priv = (NaccacheSternPrivateKeyParameters) key;
		final Vector primes = priv.getSmallPrimes();
		final Vector[] v = { plain, primes };
		return v;
	}

	/**
	 * Decrypts a BigInteger that represents the encrypted message.
	 * 
	 * @param input
	 *            The message
	 * @return A byte array that contains the decrypted message as
	 *         BigInteger.toByteArray().
	 */
	private synchronized byte[] decrypt(final BigInteger input) {
		final NaccacheSternPrivateKeyParameters priv = (NaccacheSternPrivateKeyParameters) key;
		final Vector primes = priv.getSmallPrimes();
		plain = new Vector(primes.size());
		threads = new Vector();

		// Get Chinese Remainders of CipherText
		final Object tmp = new Object();
		BigInteger smallPrime;
		Thread t;
		for (int i = 0; i < primes.size(); i++) {
			// insert objects into plain, so I can use Vector.setElementAt()
			plain.add(tmp);
			smallPrime = (BigInteger) primes.get(i);
			t = new DecryptBySmallPrime(smallPrime, input);
			threads.add(t);
		}

		final Vector runningThreads = new Vector();

		synchronized (waitFor) {

			for (int i = 0; (i < threads.size()) && (i < processorCount); i++) {
				t = (Thread) threads.get(i);
				runningThreads.add(t);
				t.start();
			}

			while (threads.size() > 0) {
				try {
					waitFor.wait();
				} catch (final InterruptedException e) {
				}
				for (int i = 0; i < threads.size(); i++) {
					t = (Thread) threads.get(i);
					if (!runningThreads.contains(t)) {
						runningThreads.add(t);
						t.start();
						break;
					}
				}
			}
		}
		for (int i = 0; i < runningThreads.size(); i++) {
			t = (Thread) runningThreads.get(i);
			try {
				t.join();
			} catch (final InterruptedException e) {
			}
		}

		// compute message from remainders
		final BigInteger retval = PrimeUtils.chineseRemainder(plain, primes);

		return retval.toByteArray();

	}

	/**
	 * Callback function for decryption threads. They submit the decrypted value
	 * modulo their prime.
	 * 
	 * @param t
	 *            The decryption thread submitting.
	 */
	private void submitDecryptionValue(final DecryptBySmallPrime t) {
		final NaccacheSternPrivateKeyParameters priv = (NaccacheSternPrivateKeyParameters) key;
		final Vector smallPrimes = priv.getSmallPrimes();
		plain.setElementAt(BigInteger.valueOf(t.lookedup), smallPrimes
				.indexOf(t.smallPrime));
		synchronized (threads) {
			threads.remove(t);
		}

		synchronized (waitFor) {
			waitFor.notifyAll();
		}
	}

	/**
	 * DecryptBySmallPrime computes the plaintext modulo a small prime t from
	 * the ciphertext. With the Chinese Remainder Theorem, the complete
	 * plaintext can be restored.
	 * 
	 * @author lippold
	 */
	class DecryptBySmallPrime extends Thread {
		private final BigInteger smallPrime;

		private final BigInteger input;

		private final NaccacheSternPrivateKeyParameters privKey;

		private int lookedup;

		private boolean debug;

		DecryptBySmallPrime(final BigInteger smallPrime, final BigInteger input) {
			super();
			this.smallPrime = smallPrime;
			this.input = input;
			privKey = (NaccacheSternPrivateKeyParameters) key;
		}

		public void run() {
			BigInteger[] pq = privKey.getPQ();
			BigInteger exponent = pq[0].subtract(BigInteger.ONE);
			exponent = exponent.multiply(pq[1].subtract(BigInteger.ONE));
			exponent = exponent.divide(smallPrime);
			Vector al = (Vector) privKey.getLookupTable().get(smallPrime);
			lookedup = al
					.indexOf(PrimeUtils.chineseModPow(input, exponent, pq));
			if (debug) {
				System.out.println("decryption for prime " + smallPrime
						+ " finished.");
			}
			submitDecryptionValue(this);
		}
	}

}
