package net.sf.distrib_rsa.cryptosystems.naccacheStern;

import java.math.BigInteger;
import java.util.Hashtable;
import java.util.Vector;

import net.sf.distrib_rsa.cryptosystems.PrimeUtils;

/**
 * Private key parameters for NaccacheStern cipher. For details on this cipher,
 * please see
 * 
 * http://www.gemplus.com/smart/rd/publications/pdf/NS98pkcs.pdf
 */
public class NaccacheSternPrivateKeyParameters extends
		NaccacheSternKeyParameters {
	
	/**
	 * @uml.property  name="pq" multiplicity="(0 -1)" dimension="1"
	 */
	private final BigInteger[] pq;
	
	/**
	 * @uml.property  name="smallPrimes"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	private final Vector smallPrimes;

	/**
	 * @uml.property  name="lookupList"
	 */
	private final Hashtable lookupList;

	/**
	 * @uml.property  name="threads"
	 * @uml.associationEnd  multiplicity="(0 -1)" inverse="this$0:de.uni_bremen.informatik.lippold.cryptosystems.naccacheStern.NaccacheSternPrivateKeyParameters$ConstructLookupVector"
	 */
	private final Vector threads;

	/**
	 * @uml.property  name="processorCount"
	 */
	private final int processorCount;

	/**
	 * @uml.property  name="debug"
	 */
	private boolean debug = false;

	/**
	 * @uml.property  name="waitFor"
	 */
	private final Object waitFor = new Object();

	/**
	 * 
	 * Constructs a NaccacheSternPrivateKey using only one CPU without debug
	 * information.
	 * 
	 * @param g
	 *            the public enryption parameter g
	 * @param n
	 *            the public modulus n = p*q
	 * @param sigma
	 *            the public sigma up to which data can be encrypted
	 * @param smallPrimes
	 *            the small primes, of which sigma is constructed in the right
	 *            order
	 * @param phi_n
	 *            the private modulus phi(n) = (p-1)(q-1)
	 * 
	 */
	public NaccacheSternPrivateKeyParameters(final BigInteger g,
			final BigInteger n, final BigInteger sigma,
			final Vector smallPrimes, final BigInteger p, final BigInteger q) {
		this(g, n, sigma, smallPrimes, p, q, false, 1);
	}

	/**
	 * Constructs a NaccacheSternPrivateKeyParameters with a given lookupList.
	 * Used by NaccacheSterKeySerializationFactory. Very fast.
	 * 
	 * @param g
	 *            public encryption parameter g
	 * @param n
	 *            public modulus n
	 * @param sigma
	 *            public sigma up to which data can be encrypted
	 * @param smallPrimes
	 *            the small primes, of which sigma is constructed in the right
	 *            order
	 * @param phi_n
	 *            the private modulus phi(n) = (p-1)(q-1)
	 * @param lookupList
	 *            a hashtable of the form [ smallPrime | Vector<BigInteger> ]
	 *            used for decryption.
	 */
	public NaccacheSternPrivateKeyParameters(final BigInteger g,
			final BigInteger n, final BigInteger sigma,
			final Vector smallPrimes, final BigInteger p, final BigInteger q,
			final Hashtable lookupList) {
		super(true, g, n, sigma);
		this.smallPrimes = smallPrimes;
		this.pq = new BigInteger[] {p, q};
		this.lookupList = lookupList;
		processorCount = 1;
		threads = null;
	}

	/**
	 * Constructs a NaccacheSternPrivateKey using as many CPU's as specified.
	 * 
	 * @param g
	 *            the public enryption parameter g
	 * @param n
	 *            the public modulus n = p*q
	 * @param sigma
	 *            the public sigma up to which data can be encrypted
	 * @param smallPrimes
	 *            the small primes, of which sigma is constructed in the right
	 *            order
	 * @param phi_n
	 *            the private modulus phi(n) = (p-1)(q-1)
	 * @param verbose
	 *            be verbose on lookupTable construction
	 * @param processorCnt
	 *            the number of threads to start. See also
	 *            Runtime.getRuntime().availableProcessors() (since Java 1.4).
	 */
	public NaccacheSternPrivateKeyParameters(final BigInteger g,
			final BigInteger n, final BigInteger sigma,
			final Vector smallPrimes, final BigInteger p, final BigInteger q,
			final boolean verbose, final int processorCnt) {
		super(true, g, n, sigma);
		this.smallPrimes = smallPrimes;
		this.pq = new BigInteger[] {p, q};
		debug = verbose;
		processorCount = processorCnt;
		lookupList = new Hashtable();
		if (debug) {
			System.out.println("current machine has " + processorCount
					+ " CPU's");
			System.out.println("Constructing lookup Array");
		}
		threads = new Vector();
		Thread t;
		BigInteger actualPrime;
		for (int i = 0; i < smallPrimes.size(); i++) {
			actualPrime = (BigInteger) smallPrimes.elementAt(i);
			t = new ConstructLookupVector(actualPrime);
			threads.add(t);
		}
		final Vector runningThreads = new Vector();

		synchronized (waitFor) {
			// Start as many threads as we have processors
			for (int i = 0; (i < processorCount) && (i < threads.size()); i++) {
				t = (Thread) threads.get(i);
				runningThreads.add(t);
				t.start();
			}
			while (threads.size() > 0) {
				try {
					waitFor.wait();
				} catch (final InterruptedException e) {
				}
				for (int i = 0; i < threads.size(); i++) {
					t = (Thread) threads.get(i);
					if (!runningThreads.contains(t)) {
						runningThreads.add(t);
						t.start();
						break;
					}
				}
			}
		}
		for (int i = 0; i < runningThreads.size(); i++) {
			t = (Thread) runningThreads.get(i);
			try {
				t.join();
			} catch (final InterruptedException e) {
			}
		}
		if (debug) {
			System.out.println("NaccacheSternPrivateKey construction finished");
		}

	}

	public BigInteger[] getPQ() {
		return pq;
	}
	
	/**
	 * @return  the smallPrimes
	 * @uml.property  name="smallPrimes"
	 */
	public Vector getSmallPrimes() {
		return smallPrimes;
	}

	public Hashtable getLookupTable() {
		return lookupList;
	}

	private void submitLookupVector(final ConstructLookupVector t) {
		synchronized (lookupList) {
			lookupList.put(t.smallPrime, t.lookup);
		}

		synchronized (threads) {
			threads.remove(t);
		}

		synchronized (waitFor) {
			waitFor.notifyAll();
		}
	}

	class ConstructLookupVector extends Thread {

		final Vector lookup = new Vector();

		final BigInteger smallPrime;

		ConstructLookupVector(final BigInteger smallPrime) {
			this.smallPrime = smallPrime;
		}

		public void run() {
			if (debug) {
				System.out.println("Constructing lookup ArrayList for "
						+ smallPrime);
			}
			BigInteger t;
			BigInteger phi = pq[0].subtract(BigInteger.ONE);
			phi = phi.multiply(pq[1].subtract(BigInteger.ONE));
			for (int j = 0; j < smallPrime.intValue(); j++) {
					t = BigInteger.valueOf(j).multiply(phi).divide(smallPrime);
					lookup.add(PrimeUtils.chineseModPow(y, t, pq));
			}
			if (debug) {
				System.out.println("thread for prime " + smallPrime
						+ " finished.");
			}
			submitLookupVector(this);
		}
	}

	public String toString() {
		String retval = super.toString();
		retval += "p:.......... " + pq[0] + "\n";
		retval += "q:.......... " + pq[1] + "\n";
		retval += "smallPrimes: " + smallPrimes + "\n";
		return retval;
	}

}
