package net.sf.distrib_rsa.ui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;

import javax.swing.JOptionPane;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.UnlockedKeyStore;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.x509.X509DefaultEntryConverter;
import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.x509.X509V3CertificateGenerator;


/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class LoadKeyStoreUI {

	private final static Logger log = Logger.getLogger(LoadKeyStoreUI.class);

	private final static String[] ksEndings = { "p12", "ks" };

	public static UnlockedKeyStore getUserKeyStore() {

		final String buttons[] = new String[2];
		buttons[0] = "Create new";
		buttons[1] = "Import existing";
		final int res = StandardDialogs.customDialogBox(
				"Do you want to create a new certificate on\n"
						+ "the fly or do you want to import a certificate?\n\n"
						+ "Note: For importing a certificate, you need\n"
						+ "a keystore with the certificate\n"
						+ "and the corresponding private key in\n"
						+ "JKS or PKCS12 format.", "Certificate needed",
				buttons);
		UnlockedKeyStore pc = null;
		if (res == 1) {
			// User wants to load existing key & cert
			log.debug("Loading KeyStore");
			final File keystore = StandardDialogs.openFileDialog(ksEndings,
					"Please select private KeyStore",
					"Please select the KeyStore with your private Key "
							+ "and your Certificate.");
			if (keystore != null) {
				boolean successful = false;
				for (int i = 0; i < 3; i++) {
					if ((pc = restoreKeys(keystore)) != null) {
						successful = true;
						break;
					}
				}
				if (!successful) {
					System.exit(1);
				}
			}
		} else {
			pc = createAndSaveNewCert();
			if (pc == null) {
				log.fatal("KeyStore not initialized, exiting");
				System.exit(1);
			}
		}
		log.debug("PrivateCertification ready");
		return pc;
	}

	private static UnlockedKeyStore restoreKeys(final File file) {
		log.debug("restoring keys from Keystore");
		try {
			KeyStore ks = null;
			final String filename = file.getName().toLowerCase();
			if (filename.endsWith("ks")) {
				ks = KeyStore.getInstance("SUN", "JKS");
			} else if (filename.endsWith("p12")) {
				ks = KeyStore.getInstance("PKCS12", "BC");
			}
			final char[] keyStorePass = StandardDialogs.passwordBox(
					"Please enter password for the selected KeyStore",
					"KeyStore password required");
			try {
				final FileInputStream fin = new FileInputStream(file);
				ks.load(fin, keyStorePass);
			} catch (final IOException ioe) {
				final String msg = ioe.toString();
				if (msg
						.matches(".*java.lang.SecurityException: Unsupported keysize or algorithm parameters.*")) {
					throw new SecurityException();
				} else {
					throw ioe;
				}
			}

			final String selectedCert = showSelectKSEntryDialog(ks,
					"Please select your certificate");
			log.debug("reading in Certificate");
			final Certificate[] certChain = ks.getCertificateChain(selectedCert);
			final X509Certificate[] cert = new X509Certificate[certChain.length];
			for (int i = 0; i < certChain.length; i++) {
				cert[i] = (X509Certificate) certChain[i];
			}

			final String selectedKey = showSelectKSEntryDialog(ks,
					"Please select your private key");
			final char[] keyPass = StandardDialogs.passwordBox(
					"Please enter password that protects your private key",
					"Key password required");
			log.debug("reading in PrivateKey");
			final PrivateKey priv = (PrivateKey) ks.getKey(selectedKey, keyPass);
			if ((cert != null) && (priv != null)) {
				final UnlockedKeyStore pc = new UnlockedKeyStore(ks, file,
						keyStorePass, selectedCert, selectedKey, keyPass);
				return pc;
			}
			return null;
		} catch (final java.lang.SecurityException e) {
			log.fatal("Caught Security Exception: ", e);
			StandardDialogs.handleSecException();
			return null;
		} catch (final Exception e) {
			log.fatal("Cannot load Keys from Keystore. Reason:\n", e);
			StandardDialogs.handleException(e);
			return null;
		}
	}

	private static UnlockedKeyStore createAndSaveNewCert() {
		final String[] strengths = { "" + 1024, "" + 2048, "" + 4096 };
		final String s = (String) JOptionPane.showInputDialog(StandardDialogs
				.getAppFrame(), "Please select desired key length",
				"Select key length", JOptionPane.PLAIN_MESSAGE, null,
				strengths, "" + 2048);
		StandardDialogs
				.showWaitDialog("Generating KeyPair (this may take a while) ...");
		final KeyPair kp = createKeyPair(Integer.parseInt(s));
		StandardDialogs.hideWaitDialog();
		UnlockedKeyStore pc = null;
		for (int i = 0; i < 3; i++) {
			pc = createKeyStore(kp);
			if (pc != null) {
				break;
			}
		}
		return pc;
	}

	private static UnlockedKeyStore createKeyStore(final KeyPair kp) {
		JOptionPane.showMessageDialog(StandardDialogs.getAppFrame(),
				"About to create a new keystore with a new certificate for you\n"
						+ "Please input the filename for the keystore first.");
		File saveTo = StandardDialogs.saveFileDialog(ksEndings,
				"Save Keystore as", "PKCS12 (.p12)");
		if (!(saveTo.getName().endsWith(".p12") || saveTo.getName().endsWith(
				"P12"))) {
			saveTo = new File(saveTo.getAbsolutePath() + ".p12");
		}
		final char[] pass = StandardDialogs.passwordBox(
				"Please input the password for your new keystore",
				"Set password for Keystore");
		final char[] pass2 = StandardDialogs.passwordBox(
				"Please confirm the password", "Confirm password");
		if (!(new String(pass)).equals(new String(pass2))) {
			StandardDialogs
					.warningBox("The passwords didn't match!\n\nStarting over...");
			log.fatal("Passwords didn't match");
			return null;
		}
		try {
			final KeyStore ks = KeyStore.getInstance("PKCS12",
					new BouncyCastleProvider());
			ks.load(null, new char[0]);
			final X509Certificate certificate[] = new X509Certificate[1];
			certificate[0] = generateCert(kp);
			final String certificateName = (String) JOptionPane
					.showInputDialog(StandardDialogs.getAppFrame(),
							"Please input a name for your certificate\n"
									+ "and your private key.", "Input name",
							JOptionPane.PLAIN_MESSAGE, null, null,
							"MyCertificate");
			final char[] keypass = StandardDialogs.passwordBox(
					"Please input the password for your private key",
					"Set password for private key");
			final char[] keypass2 = StandardDialogs.passwordBox(
					"Please confirm the password", "Confirm password");
			if (!(new String(keypass)).equals(new String(keypass2))) {
				StandardDialogs
						.warningBox("The passwords didn't match!\n\nStarting over...");
				log.fatal("Passwords didn't match");
				return null;
			}
			ks.setKeyEntry(certificateName, kp.getPrivate(), keypass,
					certificate);
			final FileOutputStream fo = new FileOutputStream(saveTo);
			try {
				ks.store(fo, pass);
			} catch (final IOException ioe) {
				final String msg = ioe.toString();
				if (msg
						.matches(".*java.lang.SecurityException: Unsupported keysize or algorithm parameters.*")) {
					throw new SecurityException();
				}
			}
			fo.close();
			return new UnlockedKeyStore(ks, saveTo, pass, certificateName,
					certificateName, keypass);
		} catch (final java.lang.SecurityException e) {
			log.fatal("Caught Security Exception: ", e);
			StandardDialogs.handleSecException();
		} catch (final Exception e) {
			log.fatal("Caught Exception", e);
			StandardDialogs.handleException(e);
		}
		return null;
	}

	private static KeyPair createKeyPair(final int bitLength) {
		log.debug("Initializing key generation");
		try {
			final SecureRandom sec_rand = EnvironmentSetup.getSecureRandom();

			final KeyPairGenerator kg = KeyPairGenerator.getInstance("RSA",
					new BouncyCastleProvider());
			kg.initialize(bitLength, sec_rand);
			log.debug("KeyGenerator inizialized");
			final KeyPair kp = kg.genKeyPair();
			log.debug("Key generated");
			return kp;
		} catch (final java.lang.SecurityException e) {
			log.fatal("Caught Security Exception:", e);
			StandardDialogs.handleSecException();
		} catch (final Exception e) {
			log.fatal("Caught Exception ", e);
			StandardDialogs.handleException(e);
		}
		return null;
	}

	private static X509Certificate generateCert(final KeyPair kp)
			throws InvalidKeyException, SecurityException, SignatureException {
		final X509V3CertificateGenerator certGen = new X509V3CertificateGenerator();
		final X509DefaultEntryConverter conv = new X509DefaultEntryConverter();
		final String asn1 = (String) JOptionPane.showInputDialog(
				StandardDialogs.getAppFrame(),
				"Please input a ASN.1 sequence describing you",
				"Input ASN.1 sequence", JOptionPane.PLAIN_MESSAGE, null, null,
				"CN=MyName,O=MyOrg,C=DE");
		final X509Name xn = new X509Name(asn1, conv);
		certGen.setIssuerDN(xn);
		certGen.setSubjectDN(xn);
		certGen.setNotBefore(new Date());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.YEAR, 2);
		certGen.setNotAfter(cal.getTime());
		certGen.setPublicKey(kp.getPublic());
		certGen.setSignatureAlgorithm("sha1WithRSAEncryption");
		certGen.setSerialNumber(new BigInteger("" + (new Date()).getTime()));
		log.debug("Generating self-signed Certificate");
		final X509Certificate cert = certGen.generateX509Certificate(kp.getPrivate());
		log.debug("Certificate generation successful");
		return cert;
	}

	private static String showSelectKSEntryDialog(final KeyStore ks,
			final String message) {
		final ArrayList choices = new ArrayList();
		Enumeration en;
		try {
			en = ks.aliases();
			while (en.hasMoreElements()) {
				choices.add(en.nextElement());
			}
		} catch (final KeyStoreException e) {
			log.error("Caught KeyStoreException: ", e);
		}
		final Object[] possibilities = choices.toArray();
		Arrays.sort(possibilities);
		final String choice = (String) JOptionPane.showInputDialog(StandardDialogs
				.getAppFrame(), message, "Select KeyStore entry",
				JOptionPane.QUESTION_MESSAGE, null, possibilities, null);
		return choice;
	}

}
