package Algorithm;

import java.util.List;

import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternPrivateKeyParameters;

import Crypto.RandomSource;
import Filter.MyBloomFilter;
import Helper.RandomString;
import Helper.Timer;
import Helper.UnitConversion;


public class BloomEncryption {

	//length of qgrams (substrings of sequence)
	private static final int q=11;
	//how many shifts are allowed
	private static final int s=20;
	//factor of how much the bloom fitler must be bigger than the size of the larger set
	private static final int lFactor=3;
	//number of hash functions to use (just for probability computation, we always use one hash function nevermind this value)
	private static final int k=1;
	private static double optimalK=1.0;
	private static double optimalL=1.0;
	//length of the bloom filter if calculated by using p, or fixed if p is not used (only relevant if l factor is not used)
	private static  long bloomLength=25000;
	//probability that a false positive occurs (to calculate l upon) (only relevant if l factor is not used)
	private static  double probability=0.01;
	private static  double probabilityLFactor=0.01;
	//bloom filter size
	private static int m;
	//number of elements in bloom filter on client side
	private static int n;
	//threshold to set for similarity measure
	private static int t;
	//maximum that the threshold can be set to
	private static int maxT;
	
	private static RandomString sequences;
	private static int sequenceLength = 1000;
	private static int editDistance = 20;
	private static int alphabetSize = 256;
	
	private static List<byte[]> encBloom;
	private static List<byte[]> encResults;
	private static MyBloomFilter<String> bloom;
	private static List<Integer> plainResults;
	
	private static final boolean usePaillier = false;
	
	private static int numRuns = 1000;
	
	public static final boolean secureRandom = false;
	
	private static final float tFactor = 0.75f;

    public static NaccacheSternKeyParameters pubParameters = null;
    public static NaccacheSternPrivateKeyParameters privParameters = null;
    
    public static final boolean doEncrypt = false;

    public static final boolean useP = true;
    public static final boolean useLFactor = false;
    public static final boolean forceEditDistance = true;

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		int i=0, j=0;
		
		if(args.length != 2 && args.length != 0) {
			System.err.println("execute with zero or two parameters for _sequence length_ and _number of runs_");
			System.exit(1);
		}
		
		if(args.length == 2) {
			sequenceLength = Integer.parseInt(args[0]);
			numRuns = Integer.parseInt(args[1]);
		}
		
		//turns to test something [edit distance from 1 to 20]
		for(j=0; j<100; j++) {
			editDistance = j+1;
			
			init();
			
			for(i=0; i<numRuns; i++) {
				runAllLocal();
				System.out.println("-----Run " + (i+1) + " out of " + numRuns + " done.-----");
			}
			
			System.out.println("Done with all " + numRuns + " runs, quitting?...");
		}
	}
	
	private static void init() {
		n = (2*s +1)*((sequenceLength+2*(q-1))-q+1);
		
		if(useP) {
			calculateBloomLength();
		} else {
			calculateProbability();
		}
		calculateProbabilityWithLFactor();
		calculateOptimalLK();

		printSettings();
	}
	
	private static void calculateOptimalLK() {
		optimalL = -1.0 * ( (n * Math.log(probability)) / Math.pow((Math.log(2)),2) );
		optimalK = optimalL/n * Math.log(2);
		
	}

	private static void calculateProbability() {
		probability = Math.pow(1.0-Math.pow(1.0-(1.0/(double)bloomLength), (double)(k*n)),(double)k);
	}

	private static void calculateProbabilityWithLFactor() {
		probabilityLFactor = Math.pow(1.0-Math.pow(1.0-(1.0/(double)(lFactor*n)), (double)(k*n)),(double)k);
	}

	private static void calculateBloomLength() {
		bloomLength = Math.round(-1.0/(Math.pow((Math.pow(-1.0*probability, 1/k) +1.0), 1.0/(k*n)) -1.0));
	}

	private static void printSettings() {
		//-----------------------------------------------------Print out settings--------------------------------------------------
		System.out.println("-----SETTINGS-----");
		System.out.println("q="+q);
		System.out.println("s="+s);
		System.out.println("k="+k);
		System.out.println("lFactor="+lFactor);
		if(useP) {
			System.out.println("probability (fixed)="+probability);
			System.out.println("estimated l="+bloomLength);
		} else {
			System.out.println("estimated probability="+probability);
			System.out.println("l (fixed)="+bloomLength);
		}
		if(useLFactor) {
			System.out.println("used l="+lFactor*n);
			System.out.println("used probability="+probabilityLFactor);
		} else {
			System.out.println("used l="+Math.round(bloomLength));
			System.out.println("estimated probability="+probability);
		}
		System.out.println("optimal l="+Math.round(optimalL));
		System.out.println("optimal k="+Math.round(optimalK));
		System.out.println("sequenceLength="+sequenceLength);
		System.out.println("elements in client set="+n);
		System.out.println("editDistance="+editDistance);
		System.out.println("alphabetSize="+alphabetSize);
		System.out.println("secureRandom="+secureRandom);
		System.out.println("doEncrypt="+doEncrypt);
		System.out.println("usePaillier="+usePaillier);
		System.out.println("numRuns="+numRuns);
		System.out.println("tFactor="+tFactor);
		System.out.println("-----SETTINGS-----");
	}

	public static void runAllLocal() {

		System.out.println("Generating random sequences.");

		//generate random sequences using size of alphabet
		sequences = new RandomString(alphabetSize);
		sequences.setSeed(RandomSource.getNextSeed(BloomEncryption.secureRandom));
		sequences.generateMainString(sequenceLength);
		sequences.generateSubStrings(1, editDistance, forceEditDistance);
		
		System.out.println("Levenshtein distance between compared strings: " + sequences.getlevenshteinDistanceToSubString(0));
		
		//-----------------------------------------------------Client's turn--------------------------------------------------
		Timer overall = new Timer();
		Client cli = runClientStart();
		
		//-----------------------------------------------------Server's turn--------------------------------------------------
		runServer(cli.getPubKeyAsByteA());
		
		//-----------------------------------------------------Client's turn--------------------------------------------------
		runClientEnd(cli);
		overall.stopPrintToLog("Total protocol time taken");
	}
	
	public static Client runClientStart() {
		
		Timer overallClient = new Timer();
		System.out.println("Initializing client.");

		Client cli = new Client(q, s);
		
		System.out.println("Set sequence and crypto system for client.");
		
		cli.init(sequences.getMainStringAsString(), usePaillier);

//		cli.printExtendedSeq();
//		srv.printExtendedSeq();

		System.out.println("Extended strings generated");

//		cli.printQGrams();
//		srv.printQGrams();

		n = cli.getNumPosQGrams();
		if(useLFactor) {
			m = lFactor * n;
		} else {
			m = Math.round(bloomLength);
		}

		cli.setBloomFilterSize(m);

		System.out.println("Bloom filters generated");
		
		cli.addQGramsToBloomFilter();

		System.out.println("Bloom filters were filled with positional q-grams");
		
		if(doEncrypt) {
			cli.encryptBloomFilter();
			System.out.println("Encrypted Bloom Filter on client side");

			overallClient.setIntermediate();
			
			byte[] pubKey = cli.getPubKeyAsByteA();
			encBloom = cli.getEncBloom();
			int size = 0;
			for(byte[] ba : encBloom) {
				size += ba.length;
			}
			size += pubKey.length;
			System.out.println("Going to transfer " + size + " bytes from client to server. " + (pubKey.length) + " key; " + (size-pubKey.length) + " data");
		}
		
		bloom = cli.getBloomFilter();
		
		return cli;
	}
	
	public static void runServer(byte[] pubKey) {
		Timer overallServer = new Timer();
		System.out.println("Initializing server.");
		
		Server srv = new Server(q, s);
		srv.init(sequences.getSubStringAsString(0), usePaillier);
		
		srv.setBloomFilterSize(m);

		maxT = srv.getNumPosQGrams();
		t=(int)(maxT*(tFactor));

		System.out.println("Threshold will start with " + t + " and move up to " + maxT);
		
		srv.addQGramsToBloomFilter();

		if(doEncrypt) {
			srv.setPubKey(pubKey);
			srv.sumEncBloom(encBloom);
	
			System.out.println("Summed up all relevant encrypted Bloom Filter elements");
		} else {
			srv.sumBloom(bloom);
		}
		
		srv.setThreshold(t);

		if(doEncrypt) {
			System.out.println("Calculate " + (maxT-t) + " encrypted results");
			srv.calcEncThresholdDiffs();
		} else {
			srv.calcPlainThresholdDiffs();
		}
		
		System.out.println("Results calculated on server side");
		overallServer.stopPrintToLog("Overall time taken by server");

		if(doEncrypt) {
			encResults = srv.getResults();
			int size = 0;
			for(byte[] ba : encResults) {
				size += ba.length;
			}
			System.out.println("Going to transfer " + size + " bytes from server to client");
		} else {
			plainResults = srv.getResultsPlain();
		}
	}
	
	public static void runClientEnd(Client cli) {
		if(doEncrypt) {
			System.out.println("Client is decrypting results...");
			cli.decryptResults(encResults);
			System.out.println("Results were decrypted by client");
		} else {
			cli.checkPlainResults(plainResults);
		}
	}

}
