package Algorithm;

import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;

import Crypto.CryptClientNS;
import Crypto.CryptClientP;
import Crypto.RandomSource;
import Filter.MyBloomFilter;
import Helper.UnitConversion;

import com.skjegstad.utils.BloomFilter;


public class Client {

	//see http://www.uniprot.org/uniprot/A7X8C2
	private final String sequence_long = "MTELKAKGPRAPHVAGGPPSPEVGSPLLCRPAAGQFPGSQTSDTLPEVSAIPISLDGLLFPRPCQGQDPSYEKTQDQQSLSDVEGAYSRAEATRGAGGSSSSPPEKESGLLDSVLDTLLAPSGPRQSQPSPPACEVTSSWSLFGPELPEDPPAAPATQGVLSPLMSRSGGKAGDSSGTAAAHKVLPQGLSPSRQLLLPASGSPHWSGAPVKPSPQPAAVEVEEEDGSESEDSAGPLLKGKPRALGGAAAGGAAAVPPGAAAGGVALVPKEDSRFSAPRVALVEQDAPMAPGRSPLATTVMDFIHVPILPLNHALLAARTRQLLEDENYDGGAGAASAFAPPRSSPSASSTPVAVGDFPDCAYPPDVEPKDDAYPLYGDFQPPALKIKEEEEGAEASARTPRSYLVAGANPAAFPDFPLGPPPPLPPRAPPSRPGEAAVTAAPASASVSSASSSGSTLECILYKAEGAPPQQGPFAPPPSKAPGAGGCLPPRDGLPSTAASASAAGAAPALYPALRLNGLPQLGYQAAVLKEGLPQVYPPYLNYLRPDSEASQSPQYSFESLPQKICLICGDEASGCHYGVLTCGSCKVFFKRAMEGQHNYLCAGRNDCIVDKIRRKNCPACRLRKCCQAGMVLGGRKFKKFNKVRVVRALDAVALPQPVGIPNESQVLSQRITFSPGQDIQLIPPLINLLMSIEPDVIYAGHDNTKPDTSSSLLTSLNQLGERQLLSVVKWSKSLPGFRNLHIDDQITLIQYSWMSLMVFGLGWRSYKHVSGQMLYFAPDLILNEQRMKESSFYSLCLTMWQIPQEFVKLQVSQEEFLCMKVLLLLNTIPLEGLRSQTQFEEMRASYIRELIKAIGLRQKGVVSSSQRFYQLTKLLDNLHDLVKQLHLYCLNTFIQSRALSVEFPEMMSEVIAAQLPKILAGMVKPLLFHKK";
	private final String sequence_short = "MTELKAKGPRAPHVAGGPPSPEVGSPLLCRPAAGQFPGSQTSDTLPEVSAIPISLDGLLFPRPCQGQDPSYEKTQDQQSLSDVEGAYSRAEATRGAGGSSSSPPEKESGLLDSVLDTLLAPSGPRQSQPSPPACEVTSSWSLFGPELPEDPPAAPATQGVLSPLMSRSGGKAGDSSGTAAAHKVL";
	private final String sequence_test = "encryption";
	private String sequence = sequence_long;
	
	private ExtendedString extSeq;
	private int q;
	private int s;
	//bloom filter size
	private int m;
	private int n;
	private int l;
	private MyBloomFilter<String> bloomFilter;
	private CryptClientP cryptoP;
	private CryptClientNS cryptoNS;
	
	private Random rnd = new RandomSource(BloomEncryption.secureRandom).getRandom();
	
	private boolean usePaillier = true;
	
	private List<byte[]> encBloom;
	
    private final boolean debug = false;

	public Client(int q, int s) {
		this.q = q;
		this.s = s;
		this.rnd.setSeed(RandomSource.getNextSeed(BloomEncryption.secureRandom));
	}
	
	public void init(boolean usePaillier) {
		this.init(null, usePaillier);
	}	
	
	public void init(String sequence, boolean usePaillier) {
		if(sequence != null)
			this.sequence = sequence;
		
		System.out.println("New extended string");
		this.extSeq = new ExtendedString(this.sequence, this.q, this.s);
		System.out.println("Generate pos. q-grams");
		this.extSeq.generatePosQGrams();
		
		this.usePaillier = usePaillier;
		
		if(usePaillier) {
			this.cryptoP = new CryptClientP();
			this.cryptoP.KeyGeneration();
		} else {
			System.out.println("New client crypt system");
			this.cryptoNS = new CryptClientNS();
			System.out.println("Init client crypt system");
			this.cryptoNS.init();
		}

		System.out.println("Client finished init. Sequence length: " + sequence.length());
	}
	
	public void printQGrams() {
		this.extSeq.printAllQGrams("Client");
	}

	public void printExtendedSeq() {
		System.out.println(this.extSeq.getExtendedSeq());
	}
	
	public int getNumPosQGrams() {
		return this.extSeq.getNumPosQGrams();
	}
	
	public void setBloomFilterSize(int n, int l) {
		this.n = n;
		this.m = n*l;
				
		this.bloomFilter = new MyBloomFilter<String>(l, n, 1);
		
		this.encBloom = new ArrayList<byte[]>();
	}

	public void setBloomFilterSize(int l) {
		this.m = l;
				
		this.bloomFilter = new MyBloomFilter<String>(l);
		
		this.encBloom = new ArrayList<byte[]>();
	}

	public void addQGramsToBloomFilter() {
		for(PosQGram pq : this.extSeq.getPosQGrams()) {
			this.bloomFilter.add(pq.getString());
		}
	}
	
	public MyBloomFilter<String> getBloomFilter() {
		return this.bloomFilter;
	}
	
	public void encryptBloomFilter() {
		int i=0;
		int size=this.bloomFilter.getBitSet().size();
		
		BigInteger r = null;
		
		if(usePaillier)
			r = new BigInteger(this.cryptoP.getBitLength(), this.rnd);
		
		System.out.println("Encrypting Bloom Filter...");
		
		for(i=0; i<size; i++) {
			if(usePaillier) {
//				this.encBloom.add(cryptoP.Encryption(toBigInt(this.bloomFilter.getBit(i)), r).toByteArray());
				this.encBloom.add(cryptoP.Encryption(this.bloomFilter.getBit(i)).toByteArray());
//				System.out.println("Encrypted element " + i + "/" + size);
			}
			else {
				this.encBloom.add(cryptoNS.encrypt(UnitConversion.boolToByteA(this.bloomFilter.getBit(i))));
//				System.out.println("Encrypted element " + i + "/" + size);
			}
		}
	}
	
	public List<byte[]> getEncBloom() {
		return this.encBloom;
	}
	
	public byte[] getPubKeyAsByteA() {
		if(usePaillier)
			return this.cryptoP.getPubKeyAsByteA();
		else
			return this.cryptoNS.getPubKeyAsByteA();
	}
	
	public void decryptResults(List<byte[]> results) {
		BigInteger ZERO = BigInteger.valueOf(0);
		BigInteger res = null;
		boolean found = false;
		int i=1;
		for(byte[] result : results) {
			if(usePaillier)
				res = this.cryptoP.Decryption(new BigInteger(1, result));
			else
				res = new BigInteger(1, this.cryptoNS.decrypt(result));
//			res = res.mod(this.crypto.getSigma());
//			System.out.println("decrypted result " + i + "/" + results.size() + " :" + res);
			if(res.compareTo(ZERO) == 0) {
				found = true;
				System.out.println("Similarity was above or equal threshold. Found after decrypting " + i + " out " + results.size() + " results.");
				break;
			}
			i++;
		}
		
		if(!found) {
			System.out.println("Similarity was below threshold.");
		}
	}

	public void checkPlainResults(List<Integer> results) {
		boolean found = false;
		int i=1;
		for(int result : results) {
			if(result == 0) {
				found = true;
				System.out.println("Similarity was above or equal threshold. Found after checking " + i + " out " + results.size() + " results.");
				break;
			}
			i++;
		}
		
		if(!found) {
			System.out.println("Similarity was below threshold.");
		}
		
		System.out.println("DEBUG: real calculated similarity is " + results.get(results.size()-1));
		System.out.println("DEBUG: maximum similarity would be " + ((Server.sequence.length()+2*(q-1))-q+1));
		System.out.println("DEBUG: distance from max is " + (((Server.sequence.length()+2*(q-1))-q+1)-results.get(results.size()-1)));
	}

//	public BigInteger getSigma() {
//		if(usePaillier)
//			return null;
//		else
//			return this.cryptoNS.getSigma();
//	}
}