package Algorithm;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeySerializationFactory;

import Crypto.CryptServerNS;
import Crypto.CryptServerP;
import Crypto.RandomSource;
import Filter.MyBloomFilter;
import Helper.UnitConversion;

import com.skjegstad.utils.BloomFilter;


public class Server {

	//see http://www.uniprot.org/uniprot/A7XW16
	private final String sequence_long = "MTELKAKGLRAPHVAGSPSSPKVGSPLPCRQATGQFPGSQTSDTLPEVSALPISLDGLLFPRICQAQDPPDEKTQDQQSLSDVEGAYSRVEATRGAGGSSSRPPEKDSGLLDSVLDTLWAPSGPGQSQPSPPACEVTSSWCLFGPELPEDPPAAPATQRVLSPLMSRSGGKAGDSSGMAAAHKVLPRGLSPSRQLLLPTSGSPHWSGAPVKPSPQPAAVEVEEEDGSESEDSAGLLLKGKPRALGGAAAGGGAPAVTPGTAAGGIALVPKEDSRFSAPRVALVEQDAPMAPGRSPLATTVTDFIHVPILPLSHALLAARTRQLLEDESYDGGSGAASAFAPPRSSPSASSTPVPGSDFPDCAYAPDAEPKDDVYPLYGDFQPPALKIKEEEEGAEASTRSPRSYLVAGASPTTFPDFPLGPPPPLPPRAPPSRPGEAAVTAAPASASVSSASSSGSTLECILYKAEGAQPQQGPFAPPPCKAPGAGGCLLPRDSLPSTSASAGATAAGAAPALYPALGLNGLPQLGYQAAVIKEGLPQVYPPYLNYLRPDSETSQSPQYSFESLPQKICLICGDEASGCHYGVLTCGSCKVFFKRAMEGQHNYLCAGRNDCIVDKIRRKNCPACRLRKCCQAGMVLGGRKFKKFNKVRVMRALDAVALPQPVGIPNENQALSQRFTFSPSQDIQLIPPLINLLLSIEPDVIYAGHDNTKPDTSSSLLTSLNQLGERQLLSVVKWSKSLPGFRNLHIDDQITLIQYSWMSLMVFGLGWRSYKHVSGQMLYFAPDLILNEQRMKESSFYSLCLTMWQIPQEFVKLQVSQEEFLCMKVLLLLNTIPLEGLRSQTQFEEMRSSYIRELIKAIGLRQKGVVSSSQRFYQLTKLLDNLHDLVKQLHLYCLNTFIQSRALSVEFPEMMSEVIAAQLPKILAGMVKPLLFHKK";
	private final String sequence_short = "MTELKAKGLRAPHVAGSPSSPKVGSPLPCRQATGQFPGSQTSDTLPEVSALPISLDGLLFPRICQAQDPPDEKTQDQQSLSDVEGAYSRVEATRGAGGSSSRPPEKDSGLLDSVLDTLWAPSGPGQSQPSPPACEVTSSWCLFGPELPEDPPAAPATQRVLSPLMSRSGGKAGDSSGMAAAHKVL";
	private final String sequence_test = "encrytion";
	public static String sequence;
	private ExtendedString extSeq;
	private int q;
	private int s;
	//bloom filter size
	private int m;
//	private int n;
	private int l;
	private NaccacheSternKeyParameters pubKey;
	private MyBloomFilter<String> bloomFilter;
	private int t;
	private int maxT;
	private CryptServerNS ctNS;
	private CryptServerP ctP;
	
	private boolean usePaillier = true;
	
	private List<byte[]> results;
	private List<Integer> resultsPlain;
	
	private byte[] encSum = null;
	private int plainSum = -1;

	public Server(int q, int s) {
		sequence = sequence_long;
		this.q = q;
		this.s = s;
		
		this.results = new ArrayList<byte[]>();
		this.resultsPlain = new ArrayList<Integer>();
	}
	
	public void init(boolean usePaillier) {
		init(null, usePaillier);
	}
	public void init(String sequence, boolean usePaillier) {
		if(sequence != null)
			this.sequence = sequence;
		this.extSeq = new ExtendedString(this.sequence, this.q, 0);
		this.extSeq.generatePosQGrams();

		this.usePaillier = usePaillier;
		
		System.out.println("Server finished init. Sequence length: " + this.sequence.length());
	}
	
	public void printQGrams() {
		this.extSeq.printAllQGrams("Server");
	}

	public void printExtendedSeq() {
		System.out.println(this.extSeq.getExtendedSeq());
	}

	public int getNumPosQGrams() {
		return this.extSeq.getNumPosQGrams();
	}
	
	public void setBloomFilterSize(int n) {
		this.m = n;
				
		this.bloomFilter = new MyBloomFilter<String>(this.m);
	}
	
	public void addQGramsToBloomFilter() {
		for(PosQGram pq : this.extSeq.getPosQGrams()) {
			this.bloomFilter.add(pq.getString());
		}
	}

	public void setPubKey(byte[] pubKey) {
		if(usePaillier) {
			this.ctP.setPubKey(pubKey);
		} else {
			BigInteger sigma = null;
			try {
				this.pubKey = NaccacheSternKeySerializationFactory.deserialize(pubKey);
				sigma = this.pubKey.getSigma();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (ClassNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			this.ctNS = new CryptServerNS(this.pubKey, sigma);
		}
	}
	
	public void sumEncBloom(List<byte[]> encBloom) {
		int i=0;
		BitSet bs = this.bloomFilter.getBitSet();
		
		BigInteger sum = null;
		encSum = null;
		
		for (i = bs.nextSetBit(0); i >= 0; i = bs.nextSetBit(i+1)) {
			if(usePaillier) {
				if(sum == null)
					sum = new BigInteger(1, encBloom.get(i));
				else {
					sum = this.ctP.cryptedAdd(sum, new BigInteger(1, encBloom.get(i)));
				}
			} else {
				if(encSum == null)
					encSum = encBloom.get(i);
				else {
					encSum = this.ctNS.addValues(encSum, encBloom.get(i));
				}
			}
		}
		if(usePaillier)
			encSum = sum.toByteArray();
	}

	public void sumBloom(MyBloomFilter<String> bloom) {
		int i=0;
		BitSet bs = this.bloomFilter.getBitSet();
		
		encSum = null;
		
		for (i = bs.nextSetBit(0); i >= 0; i = bs.nextSetBit(i+1)) {
			if(plainSum == -1)
				plainSum = UnitConversion.boolToInt(bloom.getBit(i));
			else
				plainSum += UnitConversion.boolToInt(bloom.getBit(i));
		}
	}

	public void setThreshold(int t) {
		this.t = t;
		this.maxT = this.getNumPosQGrams();
	}
		
	public void calcEncThresholdDiffs() {
		int i=0;
		byte[] tmp;
		BigInteger tmpBI;
		BigInteger rand;
		
		for(i=this.t; i<=this.maxT; i++) {
			if(usePaillier) {
				//encrypt current threshold
				tmpBI = this.ctP.Encryption(BigInteger.valueOf(i));
				
				tmpBI = this.ctP.subValues(new BigInteger(1, this.encSum), tmpBI);
				
				//multiply with random value
//				System.out.println("Generate random BigInteger mod " + this.ct.getSigma());
				rand = this.ctP.getRandomWithinModulo();
//				System.out.println("Going to multiply an encrypted value with a constant " + rand);
				tmpBI = this.ctP.cryptedMulWithConst(tmpBI, rand);
				tmp = tmpBI.toByteArray();
				this.results.add(tmp);
			} else {
				//encrypt current threshold
				tmp = this.ctNS.encrypt(UnitConversion.intToByteA(i));
				
				tmp = this.ctNS.subValues(this.encSum, tmp);
				
				//multiply with random value
//				System.out.println("Generate random BigInteger mod " + this.ct.getSigma());
				rand = this.ctNS.getRandomWithinModulo();
//				System.out.println("Going to multiply an encrypted value with a constant " + rand);
				tmp = this.ctNS.multiplyValue(tmp, rand);
				
				this.results.add(tmp);
			}
		}
		
		Random rnd = new RandomSource(BloomEncryption.secureRandom).getRandom();
		rnd.setSeed(RandomSource.getNextSeed(BloomEncryption.secureRandom)); 
		
		Collections.shuffle(this.results, rnd);
		
		//add real intersection cardinality for debug
		this.results.add(this.encSum);
	}
	
	public void calcPlainThresholdDiffs() {
		int i=0;
		int tmp;
		
		for(i=this.t; i<=this.maxT; i++) {
			tmp = this.plainSum - i;

			this.resultsPlain.add(tmp);
		}
		
		Random rnd = new RandomSource(BloomEncryption.secureRandom).getRandom();
		rnd.setSeed(RandomSource.getNextSeed(BloomEncryption.secureRandom)); 
		
		Collections.shuffle(this.resultsPlain, rnd);
		
		//add real intersection cardinality for debug
		this.resultsPlain.add(this.plainSum);
	}

	public List<byte[]> getResults() {
		return this.results;
	}
	
	public List<Integer> getResultsPlain() {
		return this.resultsPlain;
	}
}
