package Helper;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import Algorithm.BloomEncryption;
import Crypto.RandomSource;


public class RandomString {

	private static final byte DEFAULT_BITS = (byte) 8; 
	private static final int DEFAULT_ALPHABET_SIZE = 256; 
	private static final Random DEFAULT_RANDOM = new RandomSource(BloomEncryption.secureRandom).getRandom(); 

	private static final int OP_CHANGE = 0; 
	private static final int OP_INSERT = 1; 
	private static final int OP_DELETE = 2; 
	private static final int NUM_OPS = 3; 
//	private static final int DEFAULT_ = ; 
	
	private int alphabetSize = DEFAULT_ALPHABET_SIZE;
	private int strLen;
	
	private long seed = RandomSource.getNextSeed(BloomEncryption.secureRandom);
	private Random rnd;
	
	private byte[] main;
	private List<byte[]> subStrings;
	
	public RandomString() {
		this(DEFAULT_RANDOM, DEFAULT_BITS);
	}
	
	public RandomString(byte bitsPerChar) {
		this(DEFAULT_RANDOM, bitsPerChar);
	}
	
	public RandomString(Random rnd) {
		this(rnd, DEFAULT_BITS);
	}
	
	public RandomString(Random rnd, byte bitsPerChar) {
		this.rnd = rnd;

		if(bitsPerChar > 8)
			bitsPerChar = 8;
		
		if(bitsPerChar < 1)
			bitsPerChar = 1;

		this.alphabetSize = (int) Math.pow(2, bitsPerChar);

		this.subStrings = new ArrayList<byte[]>();
	}

	public RandomString(int alphabetSize) {
		this(DEFAULT_RANDOM, alphabetSize);
	}

	public RandomString(Random rnd, int alphabetSize) {
		this.rnd = rnd;
		this.alphabetSize = alphabetSize;

		this.subStrings = new ArrayList<byte[]>();
	}

	public byte[] generateMainString(int numChars) {
		int i = 0;
		
		byte[] newChars = new byte[numChars];
		this.subStrings.clear();
		
		this.rnd.nextBytes(newChars);
		
		for(i=0; i<numChars; i++) {
			newChars[i] = (byte) (newChars[i] % this.alphabetSize);
		}
		
		this.main = newChars;
		
		return this.main;
	}
	
	public void generateSubStrings(int numStrings, int numEditOps, boolean forceEditDistance) {
		int i=0,j=0,op=0;
		byte[] newStr, newStr2;
		
		LevenshteinDistance ld = new LevenshteinDistance();
		
		int pos;
		byte newChar;
		
		this.subStrings.clear();
		
		for(i=0; i<numStrings; i++) {
			newStr = this.main.clone();
			for(j=0; j<numEditOps; j++) {
				newChar = (byte) (this.rnd.nextInt() % this.alphabetSize);
				
				pos =  this.rnd.nextInt() % newStr.length;
				if (pos < 0)
				{
				    pos += newStr.length;
				}

				op = this.rnd.nextInt() % NUM_OPS; // random operation (0=change, 1=insert, 2=delete)
				if (op < 0)
				{
				    op += NUM_OPS;
				}
				
				switch(op) {
				case OP_CHANGE:
//					System.out.println("Replace on Pos:" + pos + " newChar:" + (char)newChar);
					newStr[pos] = newChar;
					break;
				case OP_INSERT:
//					System.out.println("Insert on Pos:" + pos + " newChar:" + (char)newChar);
					newStr2 = java.util.Arrays.copyOf(newStr, newStr.length+1);
					System.arraycopy(newStr, pos, newStr2, pos+1, newStr.length-pos);
					newStr2[pos] = newChar;
					newStr = newStr2;
					break;
				case OP_DELETE:
//					System.out.println("Delete on Pos:" + pos + " newChar:" + (char)newChar);
					newStr2 = java.util.Arrays.copyOf(newStr, newStr.length-1);
					System.arraycopy(newStr, pos+1, newStr2, pos, newStr.length-pos-1);
					newStr = newStr2;
					break;
				}
			}
			if(forceEditDistance && ld.LD(this.getMainStringAsString(), convertByteArrayToString(newStr, this.alphabetSize)) != numEditOps) {
				i--;
			} else {
				this.subStrings.add(newStr);
			}
		}
	}
	
	public String convertByteArrayToString(byte[] ba, int alphabetSize) {
		String str="";
		int i=0;
		
		for(i=0; i<ba.length; i++) {
			if(alphabetSize > 95) //more possible characters than printable, leave them as they are
				str += (char) (ba[i]);
			else if(alphabetSize > 26) { //more than just the 26 A-Z characters, move into printable range
				str += (char) (ba[i]+32);
			} else if(alphabetSize > 10) { //alphabet size small enough to fit into the A-Z range, so move them there, but more then 10
				str += (char) (ba[i]+65);
			} else {
				str += (char) (ba[i]+48);
			}
		}
		
		return str;
	}

	public List<byte[]> getSubStringsAsByteArray() {
		return this.subStrings;
	}
	
	public byte[] getMainStringAsByteArray() {
		return this.main;
	}
	
	public byte[] getSubStringAsByteArray(int index) {
		return this.subStrings.get(index);
	}
	
	public String[] getSubStringsAsStrings() {
		String[] ret = new String[this.subStrings.size()];
		int i = 0;
		
		
		for(byte[] ba : this.subStrings) {
			ret[i] = convertByteArrayToString(ba, this.alphabetSize);
			i++;
		}
		return ret;
	}

	public String getMainStringAsString() {
		return convertByteArrayToString(this.main, this.alphabetSize);
	}
	
	public String getSubStringAsString(int index) {
		return convertByteArrayToString(this.subStrings.get(index), this.alphabetSize);
	}


	public void setAlphabetSize(int size) {
		this.alphabetSize = size;
	}
	
	public void setBitsPerChar(byte bits) {
		if(bits > 8)
			bits = 8;
		
		if(bits < 1)
			bits = 1;

		this.alphabetSize = (int) Math.pow(2, bits);
	}
	
	public void setSeed(long seed) {
		this.seed = seed;
		this.rnd.setSeed(seed);
	}
	
	public int getlevenshteinDistanceToSubString(int n) {
		LevenshteinDistance ld = new LevenshteinDistance();
		
		return ld.LD(this.getMainStringAsString(), this.getSubStringAsString(n));
	}
}
