package net.sf.distrib_rsa;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.util.Enumeration;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class UnlockedKeyStore {

	private static final Logger log = Logger.getLogger(UnlockedKeyStore.class);

	/**
	 * @uml.property  name="ks"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.lang.String"
	 */
	final KeyStore ks;

	/**
	 * @uml.property  name="keyStorePass" multiplicity="(0 -1)" dimension="1"
	 */
	final char[] keyStorePass;

	/**
	 * @uml.property  name="certEntry"
	 */
	final String certEntry;

	/**
	 * @uml.property  name="privateKeyEntry"
	 */
	final String privateKeyEntry;

	/**
	 * @uml.property  name="privateKeyPass" multiplicity="(0 -1)" dimension="1"
	 */
	final char[] privateKeyPass;

	/**
	 * @uml.property  name="saveAs"
	 */
	final File saveAs;

	public UnlockedKeyStore(final KeyStore keyStore, final File saveAs,
			final char[] keyStorePass, final String certificateEntry,
			final String privKeyEntry, final char[] privKeyPass) {
		ks = keyStore;
		this.saveAs = saveAs;
		this.keyStorePass = keyStorePass;
		certEntry = certificateEntry;
		privateKeyEntry = privKeyEntry;
		privateKeyPass = privKeyPass;
	}

	public UnlockedKeyStore(final String keyStoreFileName,
			final String keyStoreType, final String keyStorePass,
			final String certificateName, final String keyName,
			final String keyPass) throws KeyStoreException,
			NoSuchAlgorithmException, CertificateException,
			FileNotFoundException, IOException {
		ks = KeyStore.getInstance(keyStoreType, new BouncyCastleProvider());
		saveAs = new File(keyStoreFileName);
		this.keyStorePass = keyStorePass.toCharArray();
		ks.load(new FileInputStream(saveAs), this.keyStorePass);
		certEntry = certificateName;
		privateKeyEntry = keyName;
		privateKeyPass = keyPass.toCharArray();
	}

	public PrivateKey getPrivateKey() throws KeyStoreException,
			NoSuchAlgorithmException, UnrecoverableKeyException {
		return (PrivateKey) ks.getKey(privateKeyEntry, privateKeyPass);
	}

	public Certificate[] getCertificateChain() throws KeyStoreException {
		return ks.getCertificateChain(certEntry);
	}

	public Certificate getCertificate() throws KeyStoreException {
		return ks.getCertificate(certEntry);
	}

	public void save() throws KeyStoreException, NoSuchAlgorithmException,
			CertificateException, FileNotFoundException, IOException {
		ks.store(new FileOutputStream(saveAs), keyStorePass);
	}

	public void addToProperties(final Properties prop) {
		prop.setProperty(EnvironmentSetup.KEY_STORE_FILE_NAME, saveAs
				.getAbsolutePath());
		prop.setProperty(EnvironmentSetup.KEY_STORE_PASS, new String(
				keyStorePass));
		prop.setProperty(EnvironmentSetup.KEY_STORE_TYPE, ks.getType());
		prop.setProperty(EnvironmentSetup.CERT_NAME, certEntry);
		prop.setProperty(EnvironmentSetup.KEY_NAME, privateKeyEntry);
		prop.setProperty(EnvironmentSetup.KEY_PASS, new String(privateKeyPass));
	}

	public boolean containsCertificate(final Certificate cert) {
		boolean containsMatchingEntry = false;
		Enumeration en;
		try {
			en = ks.aliases();
			while (en.hasMoreElements()) {
				final String alias = (String) en.nextElement();
				final Certificate cc = ks.getCertificate(alias);
				if (cc != null) {
					if ((new String(cc.getEncoded())).equals(new String(cert
							.getEncoded()))) {
						containsMatchingEntry = true;
						break;
					}
				}
			}
		} catch (final KeyStoreException e) {
			log.fatal("KeyStoreException", e);
		} catch (final CertificateEncodingException e) {
			// Certificates in for-loop are not encodable, should not happen
			log.fatal("CertificateEncodingException", e);
		}
		return containsMatchingEntry;
	}

	public void addCert(final String name, final Certificate cert)
			throws KeyStoreException, NoSuchAlgorithmException,
			CertificateException, FileNotFoundException, IOException {
		log.debug(ks.getProvider());
		ks.setCertificateEntry(name, cert);
		save();
	}

	public KeyStore getKeyStore() {
		return ks;
	}
}
