package net.sf.distrib_rsa.cryptosystems.benaloh;

import java.math.BigInteger;

import net.sf.distrib_rsa.cryptosystems.PrimeUtils;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator;
import org.bouncycastle.crypto.KeyGenerationParameters;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class BenalohKeyPairGenerator implements
		AsymmetricCipherKeyPairGenerator {

	/** If you want debug output, set this to true */
	private static boolean debug = false;

	/**
	 * The parameters to use. Set via init().
	 * @uml.property  name="bp"
	 * @uml.associationEnd  
	 */
	private BenalohKeyGenerationParameters bp;

	/**
	 * The prime p
	 * @uml.property  name="p"
	 */
	private BigInteger p;

	/**
	 * The prime q
	 * @uml.property  name="q"
	 */
	private BigInteger q;

	/**
	 * n = p*q
	 * @uml.property  name="n"
	 */
	private BigInteger n;

	/**
	 * gcd(y,n) =1 && y^((p-1)*(q-1)/r) mod n != 1
	 * @uml.property  name="y"
	 */
	private BigInteger y;

	/**
	 * (p-1) % r = 0 && (p-1)/r mod r != 0 && gcd(r, q-1) = 1
	 * @uml.property  name="r"
	 */
	private BigInteger r;

	/**
	 * Generates a new Benaloh KeyPair with the BenalohKeyGenerationParameters
	 * from the init() method.
	 * 
	 * @see org.bouncycastle.crypto.params.BenalohKeyGenerationParameters
	 * @see org.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator#generateKeyPair()
	 */
	public AsymmetricCipherKeyPair generateKeyPair() {
		AsymmetricCipherKeyPair retval;
		if (r.equals(BigInteger.valueOf(2))) {
			// we have a Goldwasser-Micali encryption system
			retval = generateGMKeyPair();
		} else {
			// we have a Benaloh encryption system
			retval = generateBenalohKeyPair();
		}
		return retval;
	}

	/**
	 * Generates a Benaloh key pair using the given key generation parameters.
	 * 
	 * @return An AsymmetricCipherKeyPair containing a Benaloh key pair.
	 */
	private AsymmetricCipherKeyPair generateBenalohKeyPair() {
		// Generate p
		final BigInteger startP = new BigInteger(bp.getStrength() / 2
				- r.bitLength() - 24, bp.getPrimeCertainty(), bp
				.getRandom());
		while (true) {
			p = startP.multiply(BigInteger.valueOf(2));
			p = p.multiply(new BigInteger(24, bp
					.getPrimeCertainty(), bp.getRandom()));
			p = p.multiply(r);
			p = p.add(BigInteger.ONE);
			final BigInteger pMinus1 = p.subtract(BigInteger.ONE);

			// r shall divide p-1
			if (!pMinus1.mod(r).equals(BigInteger.ZERO)) {
				if (debug) {
					System.out.println("r does not divide p-1");
				}
				continue;
			}

			// (p-1) / r and r shall be relatively prime
			if (!(r.gcd(pMinus1.divide(r)).equals(BigInteger.ONE))) {
				if (debug) {
					System.out.println("p-1 /r and r are not relatively prime");
				}
				continue;
			}

			if (!p.isProbablePrime(bp.getPrimeCertainty())) {
				if (debug) {
					System.out.println("not prime: " + p);
				}
				continue;
			}
			if (debug) {
				System.out.println("New Benaloh p " + p + " with "
						+ p.bitLength() + " bit.");
			}
			break;

		}

		// Generate q
		while (true) {
			q = new BigInteger(bp.getStrength() / 2, bp
					.getPrimeCertainty(), bp.getRandom());
			if (debug) {
				System.out.println("New Benaloh q " + q);
			}
			// r and (q-1) shall be relatively prime.
			if (!r.gcd(q.subtract(BigInteger.ONE)).equals(
					BigInteger.ONE)) {
				continue;
			}
			break;
		}
		if (debug) {
			System.out.println("New Benaloh q " + q + " with "
					+ q.bitLength() + " bit.");
		}

		n = p.multiply(q);
		if (debug) {
			System.out.println("New Benaloh n " + n + " with "
					+ n.bitLength() + " bit.");
		}

		while (true) {
			y = PrimeUtils.getRandom(n, bp.getRandom());
			if (debug) {
				System.out.println("New Benaloh y " + y);
			}

			if (!y.gcd(n).equals(BigInteger.ONE)) {
				continue;
			}
			if (y.modPow(
					((p.subtract(BigInteger.ONE)).multiply(q
							.subtract(BigInteger.ONE))).divide(r), n)
					.equals(BigInteger.ONE)) {
				continue;
			}
			break;
		}
		return new AsymmetricCipherKeyPair(new BenalohKeyParameters(y,
				r, n), new BenalohPrivateKeyParameters(y,
				r, n, p, q));

	}

	/**
	 * Generates a new Goldwasser-Micali key pair using the given key generation
	 * parameters.
	 * 
	 * @return An AsymmetricCipherKeyPair containing a Goldwasser-Micali key
	 *         pair.
	 */
	private AsymmetricCipherKeyPair generateGMKeyPair() {

		// bp are the key generation parameters supplied by the user
		p = new BigInteger(bp.getStrength() / 2, bp
				.getPrimeCertainty(), bp.getRandom());

		q = new BigInteger(bp.getStrength() / 2, bp
				.getPrimeCertainty(), bp.getRandom());

		n = p.multiply(q);

		do {
			y = PrimeUtils.getRandom(n, bp.getRandom());
		} while ((PrimeUtils.jacobiSymbol(y, p) != -1)
				&& (PrimeUtils.jacobiSymbol(y, q) != -1));

		return new AsymmetricCipherKeyPair(new BenalohKeyParameters(y,
				r, n), new BenalohPrivateKeyParameters(y,
				r, n, p, q));

	}

	/**
	 * Initializes the BenalohKeyPairGenerator. Must be called before
	 * generateKeyPair() is used.
	 * 
	 * @param param
	 *            The BenalohKeyGenerationParameters to use
	 * @see org.bouncycastle.crypto.params.BenalohKeyGenerationParameters
	 * @see org.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator#init(org.bouncycastle.crypto.KeyGenerationParameters)
	 */
	public void init(final KeyGenerationParameters param) {
		bp = (BenalohKeyGenerationParameters) param;
		r = bp.getR();
	}

	public static void setDebug(final boolean debug) {
		BenalohKeyPairGenerator.debug = debug;
	}

}
