package net.sf.distrib_rsa.cryptosystems.benaloh;

import java.math.BigInteger;

import net.sf.distrib_rsa.cryptosystems.PrimeUtils;

import org.bouncycastle.crypto.params.AsymmetricKeyParameter;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class BenalohKeyParameters extends AsymmetricKeyParameter {

	/**
	 * @uml.property  name="modulus"
	 */
	protected final BigInteger modulus;

	/**
	 * @uml.property  name="messageBorder"
	 */
	protected final BigInteger messageBorder;

	/**
	 * @uml.property  name="y"
	 */
	protected final BigInteger y;

	public BenalohKeyParameters(final BigInteger y, final BigInteger r,
			final BigInteger n) {
		this(false, y, r, n);
	}

	/**
	 * Constructor for Private/Public key
	 * 
	 * @param isPrivateKey
	 *            Is this a Private key?
	 * @param y
	 *            the public value y, an element of Z_N
	 * @param r
	 *            the public exponent r, upper limit for messages
	 * @param modulus
	 *            the public modulus
	 */
	protected BenalohKeyParameters(final boolean isPrivateKey,
			final BigInteger y, final BigInteger r, final BigInteger modulus) {
		super(isPrivateKey);
		this.modulus = modulus;
		messageBorder = r;
		this.y = y;
	}

	/**
	 * @return  the messageBorder
	 * @uml.property  name="messageBorder"
	 */
	public final BigInteger getMessageBorder() {
		return messageBorder;
	}

	/**
	 * @return  the modulus
	 * @uml.property  name="modulus"
	 */
	public final BigInteger getModulus() {
		return modulus;
	}

	/**
	 * @return  the y
	 * @uml.property  name="y"
	 */
	public final BigInteger getY() {
		return y;
	}

	public String toString() {
		final StringBuffer buf = new StringBuffer();
		buf.append("BenalohKeyParameters\n");
		buf.append("Modulus (n):............ " + modulus + "\n");
		buf.append("MessageBorder (r):...... " + messageBorder + "\n");
		buf.append("Public Encryption value: " + y + "\n");
		if (messageBorder.equals(BigInteger.valueOf(2))) {
			buf.append("J(y,n):................. "
					+ PrimeUtils.jacobiSymbol(y, modulus));
		}
		return buf.toString();
	}

}
