package net.sf.distrib_rsa.cryptosystems.benaloh;

import java.math.BigInteger;
import java.util.Hashtable;

import net.sf.distrib_rsa.cryptosystems.PrimeUtils;

/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class BenalohPrivateKeyParameters extends BenalohKeyParameters {

	/**
	 * @uml.property  name="lookupList"
	 * @uml.associationEnd  qualifier="res:java.math.BigInteger java.math.BigInteger"
	 */
	private final Hashtable lookupList;

	/**
	 * @uml.property  name="pq" multiplicity="(0 -1)" dimension="1"
	 */
	private final BigInteger[] pq;

	/**
	 * @uml.property  name="debug"
	 */
	private boolean debug = false;

	/**
	 * The default Constructor for keys generated with BenalohKeyPairGenerator.
	 * Works also for Goldwasser-Michali keys (r = 2).
	 * 
	 * @param y
	 *            the public y for encryption
	 * @param r
	 *            the public r (messageBorder)
	 * @param n
	 *            the public n (modulus)
	 * @param p
	 *            the private prime p
	 * @param q
	 *            the private prime q
	 */
	public BenalohPrivateKeyParameters(final BigInteger y, final BigInteger r,
			final BigInteger n, final BigInteger p, final BigInteger q) {
		super(true, y, r, n);
		pq = new BigInteger[] {p,q};
		lookupList = new Hashtable();
	}

	/**
	 * Constructs the lookupList for decryption. Only needed for Benaloh
	 * systems. If not generated during key setup, it will be generated on the
	 * fly during decryption.
	 * 
	 * Uses CRT for faster generation.
	 */
	public void setupLookupList() {
		final BigInteger pSub1 = pq[0].subtract(BigInteger.ONE);
		final BigInteger qSub1 = pq[1].subtract(BigInteger.ONE);
		BigInteger t;
		BigInteger res;
		for (BigInteger i = BigInteger.ZERO; i.compareTo(super.messageBorder) < 0; i = i
				.add(BigInteger.ONE)) {
			if (debug) {
				System.out.println("Constructing lookupValue for " + i);
			}
			t = i.multiply(pSub1.multiply(qSub1).divide(messageBorder));
			res = PrimeUtils.chineseModPow(y, t, pq);
			lookupList.put(res, i);
		}

	}

	/**
	 * Constructor for pre-generated _true_ Benaloh Keys. This does not work
	 * with Goldwasser-Michali keys (messageBorder = r = 2).
	 * 
	 * @param y
	 *            the Y
	 * @param r
	 *            the messageBorder, also known as r
	 * @param n
	 *            the modulus, also known as n = p*q
	 * @param phi_n
	 *            the phi_n = (p-1)(q-1)
	 * @param lookup
	 *            the pre-computed lookup table
	 */
	public BenalohPrivateKeyParameters(final BigInteger y, final BigInteger r,
			final BigInteger n, final BigInteger p, final BigInteger q, final Hashtable lookup) {
		super(true, y, r, n);
		pq = new BigInteger[] {p,q};
		lookupList = lookup;
	}

	/**
	 * @return  the lookupList
	 * @uml.property  name="lookupList"
	 */
	public final Hashtable getLookupList() {
		return lookupList;
	}

	/**
	 * @return the pq[]
	 */
	public final BigInteger[] getPQ() {
		return pq;
	}

	/**
	 * @param debug  the debug to set
	 * @uml.property  name="debug"
	 */
	public void setDebug(final boolean debug) {
		this.debug = debug;
	}

	public String toString() {
		final StringBuffer buf = new StringBuffer();
		buf.append(super.toString() + "\n");
		if (super.messageBorder.equals(BigInteger.valueOf(2))) {
			buf.append("p:...................... " + pq[0] + "\n");
			buf.append("y^((p-1)/2) mod p:...... "
					+ super.y.modPow(pq[0].subtract(BigInteger.ONE).divide(
							BigInteger.valueOf(2)), pq[0]) + "\n");
			buf.append("J(y,p):................. "
					+ PrimeUtils.jacobiSymbol(y, pq[0]) + "\n");
			buf.append("q:...................... " + pq[1] + "\n");
			buf.append("y^((q-1)/2) mod q:...... "
					+ super.y.modPow(pq[1].subtract(BigInteger.ONE).divide(
							BigInteger.valueOf(2)), pq[1]) + "\n");
			buf.append("J(y,q):................. "
					+ PrimeUtils.jacobiSymbol(y, pq[1]) + "\n");
		} else {
			buf.append("p:...................... " + pq[0]);
			buf.append("q:...................... " + pq[1]);
		}
		return buf.toString();
	}
}
