package net.sf.distrib_rsa.cryptosystems.keyGeneration;

import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.cryptosystems.PrimeUtils;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyGenerationParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyPairGenerator;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeySerializationFactory;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternPrivateKeyParameters;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.util.encoders.Base64;


/**
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class CreateNaccacheSternKeyPair {

	/**
	 * Creates a NaccacheSternKeyPair that can encrypt up to keySize bits in one
	 * pass.
	 * 
	 * @param args
	 *            <ul>
	 *            <li> [-sigmaSize bits] the desired size of Sigma </li>
	 *            <li> [-debug] turn on debugging information </li>
	 *            <li> [-pubOut filename] save public key to filename </li>
	 *            <li> [-privOut filename] save private key to filename </li>
	 *            <li> [-certainty int] Certainty that primes are really prime
	 *            is then (1 - (1/2^certainty))</li>
	 *            </ul>
	 * @throws NoSuchProviderException
	 * @throws NoSuchAlgorithmException
	 * @throws IOException
	 */
	public static void main(final String[] args) throws NoSuchAlgorithmException,
			NoSuchProviderException, IOException {

		System.out.println("running on "
				+ Runtime.getRuntime().availableProcessors() + "CPU's");
		int sigmaSize = 4096;
		boolean debug = false;
		String pubOut = "Public " + sigmaSize + " Bit sigma Key.txt";
		String privOut = "Private " + sigmaSize + " Bit sigma Key.txt";
		int certainty = 40;

		// parse command line args
		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("-sigmaSize")) {
				sigmaSize = Integer.parseInt(args[i + 1]);
			}
			if (args[i].equals("-debug")) {
				debug = true;
			}
			if (args[i].equals("-pubOut")) {
				pubOut = args[i + 1];
			}
			if (args[i].equals("-privOut")) {
				privOut = args[i + 1];
			}
			if (args[i].equals("-certainty")) {
				certainty = Integer.parseInt(args[i + 1]);
			}
		}
		final AsymmetricCipherKeyPair pair = getNCSKeyPair(sigmaSize, certainty,
				debug);

		final NaccacheSternPrivateKeyParameters priv = (NaccacheSternPrivateKeyParameters) pair
				.getPrivate();
		final NaccacheSternKeyParameters pub = (NaccacheSternKeyParameters) pair
				.getPublic();
		final FileOutputStream pubFile = new FileOutputStream(pubOut);
		pubFile.write(Base64.encode(NaccacheSternKeySerializationFactory
				.getSerialized(pub)));
		pubFile.close();
		final FileOutputStream privFile = new FileOutputStream(privOut);
		privFile.write(Base64.encode(NaccacheSternKeySerializationFactory
				.getSerialized(priv)));
		privFile.close();
	}

	public static AsymmetricCipherKeyPair getNCSKeyPair(final int sigmaSize,
			final int certainty, final boolean debug) throws NoSuchAlgorithmException,
			NoSuchProviderException {

		final SecureRandom random = EnvironmentSetup.getSecureRandom();

		// NacSt-Cipher has to be harder than our key. It is imperative that
		// keySize < sigma.bitLength. So we have to make Sigma big enough.
		// NacSt'98 recommends n >> sigma^4:

		BigInteger sigma = BigInteger.ONE;
		int noOfSmallPrimes = 0;
		final int[] sortedSmallPrimes = PrimeUtils.getFirstPrimes(sigmaSize / 2);
		if (debug) {
			System.out.println(sortedSmallPrimes.length);
		}
		while ((sigma.bitLength() <= sigmaSize)
				&& (noOfSmallPrimes < sortedSmallPrimes.length)) {
			// start with 3, not with 2
			sigma = sigma.multiply(BigInteger
					.valueOf(sortedSmallPrimes[noOfSmallPrimes + 1]));
			noOfSmallPrimes++;
		}
		if (sigma.bitLength() <= sigmaSize) {
			final String errorTxt = "Didn't have enough smallPrimes in"
					+ " NaccacheSternKeyGenerationParameters for desired keySize";
			System.out.println(errorTxt);
			System.exit(1);
		}

		if (noOfSmallPrimes % 2 == 1) {
			noOfSmallPrimes++;
		}

		// In section 3.1 of NaccacheStern98 they state that |sigma|/|n| < 1/4.
		// So let's adjust n appropriatley:
		final NaccacheSternKeyGenerationParameters param = new NaccacheSternKeyGenerationParameters(
				random, sigma.bitLength() * 4 + 1, certainty, noOfSmallPrimes,
				debug);

		// Generate new KeyPair with as many Processors as possible
		final NaccacheSternKeyPairGenerator keyGen = new NaccacheSternKeyPairGenerator(
				Runtime.getRuntime().availableProcessors());
		keyGen.init(param);
		final AsymmetricCipherKeyPair pair = keyGen.generateKeyPair();
		return pair;
	}
}
