/**
 * 
 */
package net.sf.distrib_rsa.protocols.computeD;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;

import org.apache.log4j.Logger;

/**
 * @author lippold
 * 
 */
public abstract class ComputeKey extends Thread {

	private static final Logger log = Logger.getLogger(ComputeKey.class);

	/**
	 * @uml.property  name="p"
	 */
	final protected BigInteger p;

	/**
	 * @uml.property  name="q"
	 */
	final protected BigInteger q;

	/**
	 * @uml.property  name="rsaN"
	 */
	final protected BigInteger rsaN;

	/**
	 * @uml.property  name="multShare_Phi_n_mod_e"
	 */
	protected BigInteger multShare_Phi_n_mod_e;

	/**
	 * @uml.property  name="exp"
	 */
	protected BigInteger exp;

	/**
	 * @uml.property  name="phi"
	 */
	protected BigInteger phi;

	/**
	 * @uml.property  name="psi"
	 */
	protected BigInteger psi;

	/**
	 * @uml.property  name="zeta"
	 */
	protected BigInteger zeta;

	/**
	 * @uml.property  name="d"
	 */
	protected BigInteger d;

	/**
	 * @uml.property  name="alpha1"
	 */
	protected BigInteger alpha1;

	/**
	 * @uml.property  name="alpha2"
	 */
	protected BigInteger alpha2;

	/**
	 * @uml.property  name="ring"
	 */
	protected BigInteger ring;

	/**
	 * @uml.property  name="rEMOTE_IP"
	 */
	protected final String REMOTE_IP;

	protected final static String PROTO_VIOL = "Protocol violation. Not executing ";

	/**
	 * @uml.property  name="out"
	 */
	protected final ObjectOutputStream out;

	/**
	 * @uml.property  name="in"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	protected final ObjectInputStream in;

	/**
	 * @uml.property  name="rand"
	 */
	protected final SecureRandom rand;

	/**
	 * @uml.property  name="running"
	 */
	protected boolean running = true;

	/**
	 * @uml.property  name="pubKey"
	 */
	protected X509EncodedKeySpec pubKey;

	/**
	 * @uml.property  name="privKey"
	 */
	protected PKCS8EncodedKeySpec privKey;

	/**
	 * @uml.property  name="successful"
	 */
	boolean successful = true;

	protected static final String[] states = { "PUB_EXP", "PHI_MOD_EXP",
			"OBLIVIOUS_TRANSFER", "OBL_TRANSFER_FINISHED", "Y_ADD_PHI_R",
			"COMPUTE_PSI", "COMPUTE_PSI_FINISHED", "Y_SUB_PSI",
			"COMPUTE_ALPHA", "ALPHA1", "ALPHA2", "COMPUTE_D" };

	/**
	 * @uml.property  name="actualState"
	 */
	protected String actualState;

	public ComputeKey(final BigInteger p, final BigInteger q,
			final BigInteger rsaN, final ObjectInputStream in,
			final ObjectOutputStream out, final String remoteIP,
			final SecureRandom rand) throws IOException,
			NoSuchAlgorithmException, NoSuchProviderException {
		log.debug("initializing ComputeKey");
		REMOTE_IP = remoteIP;
		this.out = out;
		this.in = in;
		this.rand = rand;
		this.p = p;
		this.q = q;
		this.rsaN = rsaN;
		log.debug("ComputeKey initialized");
	}

	public void endThread() {
		running = false;
	}

	/**
	 * @return  the actualState
	 * @uml.property  name="actualState"
	 */
	protected String getActualState() {
		return actualState;
	}

	/**
	 * Gets the public exponent e
	 * @return  the exp
	 * @uml.property  name="exp"
	 */
	public BigInteger getExp() {
		return exp;
	}

	/**
	 * @return  the multShare_Phi_n_mod_e
	 * @uml.property  name="multShare_Phi_n_mod_e"
	 */
	protected BigInteger getMultShare_Phi_n_mod_e() {
		return multShare_Phi_n_mod_e;
	}

	/**
	 * @return  the phi
	 * @uml.property  name="phi"
	 */
	protected BigInteger getPhi() {
		return phi;
	}

	/**
	 * @return  the psi
	 * @uml.property  name="psi"
	 */
	protected BigInteger getPsi() {
		return psi;
	}

	/**
	 * @return  the zeta
	 * @uml.property  name="zeta"
	 */
	protected BigInteger getZeta() {
		return zeta;
	}

	/**
	 * Gets the private exponent d
	 * @return  the d
	 * @uml.property  name="d"
	 */
	public BigInteger getD() {
		return d;
	}

	/**
	 * @return  the alpha1
	 * @uml.property  name="alpha1"
	 */
	protected BigInteger getAlpha1() {
		return alpha1;
	}

	/**
	 * @return  the alpha2
	 * @uml.property  name="alpha2"
	 */
	protected BigInteger getAlpha2() {
		return alpha2;
	}

	/**
	 * @return  the ring
	 * @uml.property  name="ring"
	 */
	protected BigInteger getRing() {
		return ring;
	}

	/**
	 * @return  the privKey
	 * @uml.property  name="privKey"
	 */
	public PKCS8EncodedKeySpec getPrivKey() {
		return privKey;
	}

	/**
	 * @return  the pubKey
	 * @uml.property  name="pubKey"
	 */
	public X509EncodedKeySpec getPubKey() {
		return pubKey;
	}

	/**
	 * @return the successful
	 */
	public boolean wasSuccessful() {
		return successful;
	}

	protected void generateRSAKey() {
		log.info(REMOTE_IP + "finished RSA key generation");
		log.debug(REMOTE_IP + "Generated new shared RSA key:");
		log.debug(REMOTE_IP + "Public modulus N:   " + rsaN);
		log.debug(REMOTE_IP + "Public exponent e:  " + exp);
		log.debug(REMOTE_IP + "Private exponent d: " + d);

		final RSAPrivateKeySpec privKeySpec = new RSAPrivateKeySpec(rsaN,
				getD());
		final RSAPublicKeySpec pubKeySpec = new RSAPublicKeySpec(rsaN, getExp());
		Provider[] providers = Security.getProviders();
		boolean successful = false;
		for (int i = 0; i < providers.length; i++) {
			try {
				KeyFactory keyFactory = KeyFactory.getInstance("RSA",
						providers[i]);
				final Key livePrivKey = keyFactory.generatePrivate(privKeySpec);
				final Key livePubKey = keyFactory.generatePublic(pubKeySpec);
				pubKey = (X509EncodedKeySpec) keyFactory.getKeySpec(livePubKey,
						X509EncodedKeySpec.class);
				privKey = (PKCS8EncodedKeySpec) keyFactory.getKeySpec(
						livePrivKey, PKCS8EncodedKeySpec.class);
				successful = true;
			} catch (final Exception e) {
				log.debug("Could not convert keys to PKCS#8 "
						+ "data, trying next provider", e);
			}
			if (successful) {
				break;
			}
		}
		if (!successful) {
			log.fatal(REMOTE_IP + "Could not generate encoded Keys "
					+ "since no provider was able to.");
			log.info(REMOTE_IP + "You will have to generate it "
					+ "elsewhere, the needed parameters are:");
			log.info(REMOTE_IP + "Public modulus N:   " + rsaN);
			log.info(REMOTE_IP + "Public exponent e:  " + exp);
			log.info(REMOTE_IP + "Private exponent d: " + d);
			log.info(REMOTE_IP + "Usually, encoded keys are available"
					+ " for N >= 512 bit.");
		}
	}
}
