/**
 * 
 */
package net.sf.distrib_rsa.protocols.gilboaSharing;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.util.Hashtable;
import java.util.Vector;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.cryptosystems.PrimeUtils;
import net.sf.distrib_rsa.protocols.RSAParameters;

import org.apache.log4j.Logger;

/**
 * @author lippold
 * 
 */
public abstract class GilboaCommon extends Thread implements
		RSAParameters {

	final private static Logger log = Logger
			.getLogger(GilboaCommon.class);

	final protected static String[] states = {
			"EXCHANGE_KEY_SIZE",
			"START_BENALOH_SETUP", // not used any more
			"BENALOH_SETUP", "BENALOH_SETUP_FINISHED", "CANDIDATE_GENERATION",
			"CANDIDATE_FOUND", "CANDIDATE_VERIFICATION", "CANDIDATE_VERIFIED",
			"NEW_N", "PREPARE_BIPRIME_TEST", "RSA_KEY_GENERATION", "AGREE_ON_E", "" };

	/**
	 * @uml.property  name="actualState"
	 */
	protected String actualState;

	final protected static String PROTO_VIOL = "Protocol violation. Not executing ";

	/**
	 * @uml.property  name="out"
	 */
	final protected ObjectOutputStream out;

	/**
	 * @uml.property  name="in"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected ObjectInputStream in;

	/**
	 * @uml.property  name="running"
	 */
	protected boolean running = true;

	/**
	 * @uml.property  name="preferredSharedKeySize"
	 */
	protected int preferredSharedKeySize = EnvironmentSetup.getDesiredKeySize();

	/**
	 * @uml.property  name="primeCertainty"
	 */
	protected int primeCertainty = EnvironmentSetup.getPrimeCertainty();

	/**
	 * @uml.property  name="genPrimes"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected Vector genPrimes = new Vector();

	/**
	 * @uml.property  name="verPrimes"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected Vector verPrimes = new Vector();

	/**
	 * @uml.property  name="pComposites"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected Vector pComposites = new Vector();

	/**
	 * @uml.property  name="qComposites"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected Vector qComposites = new Vector();

	/**
	 * @uml.property  name="benalohSystems"
	 * @uml.associationEnd  qualifier="prime:java.math.BigInteger de.uni_bremen.informatik.lippold.cryptosystems.benaloh.BenalohSystem"
	 */
	final protected Hashtable benalohSystems = new Hashtable();

	/**
	 * @uml.property  name="rand"
	 */
	final protected SecureRandom rand;

	/**
	 * @uml.property  name="rsaN"
	 */
	protected BigInteger rsaN;

	/**
	 * @uml.property  name="rEMOTE_IP"
	 */
	protected final String REMOTE_IP;

	/**
	 * @uml.property  name="genProd"
	 */
	BigInteger genProd = BigInteger.ONE;

	/**
	 * @uml.property  name="p"
	 */
	protected BigInteger p;

	/**
	 * @uml.property  name="q"
	 */
	protected BigInteger q;

	/**
	 * @uml.property  name="four"
	 */
	protected final BigInteger four = BigInteger.valueOf(4);

	/**
	 * @uml.property  name="minPrimes" multiplicity="(0 -1)" dimension="1"
	 */
	private int[] minPrimes;

	/**
	 * @uml.property  name="o"
	 */
	private final Object o = new Object();

	/**
	 * @uml.property  name="successful"
	 */
	protected boolean successful = true;

	/**
	 * @uml.property  name="nMod4"
	 */
	protected BigInteger nMod4;

	/**
	 * Superclass for distributed Sieving
	 * 
	 * @param socket
	 *            the socket this thread communicates over
	 * @throws IOException
	 *             if the streams cannot be constructed from the socket
	 * @throws NoSuchProviderException
	 * @throws NoSuchAlgorithmException
	 */
	public GilboaCommon(final String remoteIP,
			final ObjectInputStream in, final ObjectOutputStream out)
			throws IOException, NoSuchAlgorithmException,
			NoSuchProviderException {
		REMOTE_IP = remoteIP;
		this.out = out;
		this.in = in;
		rand = EnvironmentSetup.getSecureRandom();
		minPrimes = new int[0];
	}

	public void endThread() {
		running = false;
	}

	protected void populatePrimeLists() throws NoSuchAlgorithmException,
			NoSuchProviderException {
		log.debug("populating prime lists");

		// fetch the first n primes. We cannot do this in the constructor,
		// since the parties first need to agree on the keysize.
		boolean initialized = true;
		if (minPrimes.length == 0) {
			minPrimes = PrimeUtils.getFirstPrimes(preferredSharedKeySize);
			initialized = false;
		}

		int primeCnt = 0;

		// From [Gilboa 99]: "Let p_1, ... p_j' be the smallest j' distinct
		// primes such that P[i=1, j] p_i > 2^((sigma-1)/2)." Alice and Bob pick
		// their elements at random for each p_i. Then, with crt, they get their
		// final p and q.
		if (initialized) {
			for (int i = 0; i < genPrimes.size(); i++) {
				pComposites.set(i, o);
				qComposites.set(i, o);
			}
		} else {
			genProd = BigInteger.ONE;
			BigInteger prime;
			do {
				int p = minPrimes[primeCnt];
				if (p == 2) {
					// replace 2 by 4 so that the sharings keep their congruency
					// mod 4. 4 is also relatively prime to all other primes :)
					p = 4;
				}
				if (p == 3) {
					// Don't use 3, we cannot set it correctly because it is
					// always 0
					primeCnt++;
					continue;
				}
				prime = BigInteger.valueOf(p);
				genProd = genProd.multiply(prime);
				genPrimes.add(prime);
				pComposites.add(o);
				qComposites.add(o);
				primeCnt++;
			} while (genProd.bitLength() < preferredSharedKeySize / 2);

			log.debug("Generating Primes have " + genPrimes.size()
					+ " primes, the largest prime is "
					+ minPrimes[primeCnt - 1]
					+ ", the product of all Primes is " + genProd + " with "
					+ genProd.bitLength() + " bit.");

			BigInteger verProd = genProd.abs();
			verPrimes.removeAllElements();
			do {
				prime = BigInteger.valueOf(minPrimes[primeCnt]);
				verProd = verProd.multiply(prime);
				verPrimes.add(prime);
				primeCnt++;
			} while (verProd.bitLength() < genProd.bitLength() * 2);

			log.debug("Verifying Primes have " + verPrimes.size()
					+ " primes, the largest prime is "
					+ minPrimes[primeCnt - 1]
					+ ", the product of all Primes is " + verProd + " with "
					+ verProd.bitLength() + " bit.");
		}

	}

	/**
	 * @return  the p
	 * @uml.property  name="p"
	 */
	public BigInteger getP() {
		return p;
	}

	/**
	 * @return  the q
	 * @uml.property  name="q"
	 */
	public BigInteger getQ() {
		return q;
	}

	/**
	 * @return  the rsaN
	 * @uml.property  name="rsaN"
	 */
	public BigInteger getRsaN() {
		return rsaN;
	}

	/**
	 * @return the successful
	 */
	public boolean wasSuccessful() {
		return successful;
	}

	protected BigInteger[] generateNewPrimes(final BigInteger prime) {
//		final int pos = genPrimes.indexOf(prime);
//		final Vector primes = new Vector();
//		final Vector pModuli = new Vector();
//		final Vector qModuli = new Vector();
//		for (int i = 0; i < pos; i++) {
//			primes.add(genPrimes.get(i));
//			pModuli.add(pComposites.get(i));
//			qModuli.add(qComposites.get(i));
//		}
//		BigInteger oldP;
//		BigInteger oldQ;
//		log.info("constructing remainder with " + prime
//				+ " and vector primes: " + primes + " and pModuli " + pModuli
//				+ " and q Moduli " + qModuli);
//		if (primes.size() > 0) {
//			oldP = PrimeUtils.chineseRemainder(pModuli, primes);
//			oldQ = PrimeUtils.chineseRemainder(qModuli, primes);
//		} else {
//			if (nMod4.equals(BigInteger.valueOf(3))) {
//				oldP = BigInteger.valueOf(2);
//				oldQ = BigInteger.valueOf(2);
//			} else {
//				oldP = BigInteger.ZERO;
//				oldQ = BigInteger.ZERO;
//			}
//		}
		BigInteger addP;
//		do {
			addP = PrimeUtils.getRandom(prime, rand);
//		} while (!addP.add(oldP).mod(four).equals(nMod4));

		BigInteger addQ;
//		do {
			addQ = PrimeUtils.getRandom(prime, rand);
//		} while (!addQ.add(oldQ).mod(four).equals(nMod4));
		final BigInteger[] retval = { addP, addQ };
		return retval;
	}
}
