package net.sf.distrib_rsa.protocols.gilboaSharing;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.Vector;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.cryptosystems.PrimeUtils;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohEngine;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohKeyParameters;
import net.sf.distrib_rsa.cryptosystems.benaloh.BenalohSystem;

import org.apache.log4j.Logger;
import org.bouncycastle.crypto.digests.SHA1Digest;


/**
 * GilboaServer is the thread on the server that is started for every connecting
 * client. It holds a private p and q that are both equal to 3 mod 4. This eases
 * computation of N, because the client just guesses his p and q in a way that
 * (client_p + server_p)(client_q + server_q) = N = prime.
 * 
 * @author lippold Published under the GPLv2 Licence (c) 2006 Georg Lippold
 * 
 */
public class GilboaServer extends GilboaCommon {

	private static final Logger log = Logger.getLogger(GilboaServer.class);

	/**
	 * @uml.property name="computedZs"
	 * @uml.associationEnd multiplicity="(0 -1)"
	 *                     elementType="java.math.BigInteger"
	 */
	private final Vector computedZs = new Vector();

	/**
	 * @uml.property name="primeTries"
	 * @uml.associationEnd multiplicity="(0 -1)"
	 *                     elementType="java.math.BigInteger"
	 */
	private Vector primeTries;

	/**
	 * @uml.property name="tries"
	 */
	int tries = 0;

	/**
	 * @uml.property name="benProd"
	 */
	private BigInteger benProd = BigInteger.ONE;

	/**
	 * @param socket
	 * @throws IOException
	 * @throws NoSuchProviderException
	 * @throws NoSuchAlgorithmException
	 */
	public GilboaServer(final String remIP, final ObjectInputStream in,
			final ObjectOutputStream out) throws IOException,
			NoSuchAlgorithmException, NoSuchProviderException {
		super(remIP, in, out);
		nMod4 = BigInteger.valueOf(3);
	}

	public void run() {
		log.info(REMOTE_IP + "Starting distributed sieving");
		int command = -1;
		try {
			while (running && ((command = in.read()) != -1)) {

				switch (command) {

				case GilboaProtocol.PREFERRED_SHARED_KEY_SIZE:
					if (super.actualState == null) {
						log.debug(REMOTE_IP + "Got PREFERRED_SHARED_KEY_SIZE");
						super.actualState = states[0];
						agreeOnPrimeLength();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "PREFERRED_SHARED_KEY_SIZE");
					}
					break;

				case GilboaProtocol.START_BENALOH_SETUP:
					if (super.actualState.equals(states[0])) {
						log.debug(REMOTE_IP + "Got START_BENALOH_SETUP");
						super.actualState = states[2];
						out.write(GilboaProtocol.START_BENALOH_SETUP);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "START_BENALOH_SETUP");
					}
					break;

				case GilboaProtocol.BENALOH_SETUP:
					if (super.actualState.equals(states[2])) {
						log.debug(REMOTE_IP + "Got BENALOH_SETUP");
						readBenalohSystem();
						out.write(GilboaProtocol.BENALOH_SETUP);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "BENALOH_SETUP");
					}
					break;

				case GilboaProtocol.BENALOH_SETUP_FINISHED:
					if (actualState.equals(states[2])) {
						log.debug(REMOTE_IP + "Got BENALOH_SETUP_FINISHED");
						actualState = states[3];
						super.populatePrimeLists();
						if (benalohSystems.size() >= (genPrimes.size() + verPrimes
								.size())
								&& benProd.bitLength() >= preferredSharedKeySize) {
							log.debug(REMOTE_IP + "writing "
									+ "BENALOH_SETUP_FINISHED ( = success)");
							out.write(GilboaProtocol.BENALOH_SETUP_FINISHED);
							out.flush();
						} else {
							log.fatal(REMOTE_IP + "Expected "
									+ (genPrimes.size() + verPrimes.size())
									+ " Benaloh systems," + " but got only "
									+ benalohSystems.size());
							log.fatal(REMOTE_IP
									+ "Not all necessary Benaloh Systems "
									+ "were transferred.\n"
									+ "Writing BENALOH_SETUP_FAILED.");
							out.write(GilboaProtocol.BENALOH_SETUP_FAILED);
							out.flush();
							log.fatal(REMOTE_IP + "Exiting.");
							successful = false;
							endThread();
						}
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "BENALOH_SETUP_FINISHED");
					}
					break;

				case GilboaProtocol.START_CANDIDATE_GENERATION:
					if (super.actualState.equals(states[3])) {
						log.debug(REMOTE_IP + "Got START_CANDIDATE_GENERATION");
						tries++;
						// System.runFinalization();
						// System.gc();
						// Thread.yield();
						// generatePQ();
						prepareZVector();
						super.actualState = states[4];
						log.debug(REMOTE_IP
								+ "Writing START_CANDIDATE_GENERATION");
						out.write(GilboaProtocol.START_CANDIDATE_GENERATION);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "START_CANDIDATE_GENERATION");
					}
					break;

				case GilboaProtocol.CONVERSION_MOD_PRIME:
					log.debug(REMOTE_IP + "GOT CONVERSION_MOD_PRIME");
					if (super.actualState.equals(states[4])
							|| super.actualState.equals(states[6])) {
						final BigInteger prime = (BigInteger) in.readObject();
						computeN_(prime);
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "CONVERSION_MOD_PRIME");
					}
					break;

				case GilboaProtocol.GENERATION_FINISHED:
					log.debug(REMOTE_IP + "Got GENERATION_FINISHED");
					if (super.actualState.equals(states[4])) {
						super.actualState = states[5];
						if (verifyGeneration()) {
							out.write(GilboaProtocol.GENERATION_FINISHED);
							out.flush();
						} else {
							out.write(GilboaProtocol.N_FAILED);
							out.flush();
						}
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "GENERATION_FINISHED");
					}
					break;

				case GilboaProtocol.START_VERIFICATION:
					log.debug(REMOTE_IP + "Got START_VERIFICATION");
					if (super.actualState.equals(states[5])) {
						super.actualState = states[6];
						out.write(GilboaProtocol.START_VERIFICATION);
						out.flush();
					} else {
						log
								.fatal(REMOTE_IP + PROTO_VIOL
										+ "START_VERIFICATION");
					}
					break;

				case GilboaProtocol.VERIFICATION_FINISHED:
					log.debug(REMOTE_IP + "Got VERIFICATION_FINISHED");
					if (super.actualState.equals(states[6])) {

						super.actualState = states[7];
						log.info(REMOTE_IP
								+ "Verification finished successfully");
						out.write(GilboaProtocol.VERIFICATION_FINISHED);
						out.flush();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "VERIFICATION_FINISHED");
					}
					break;

				case GilboaProtocol.NEW_N:
					log.debug(REMOTE_IP + "Got NEW_N");
					if (super.actualState.equals(states[7])) {
						super.actualState = states[8];
						rsaN = (BigInteger) in.readObject();
						log.debug(REMOTE_IP + "Computed N is " + rsaN);
						if (rsaN.bitLength() < super.preferredSharedKeySize) {
							log.fatal(REMOTE_IP
									+ "Generated N is smaller than "
									+ preferredSharedKeySize);
							log.fatal(REMOTE_IP + "Complaining.");
							out.write(GilboaProtocol.N_FAILED);
							super.actualState = states[3];
							out.flush();
							break;
						} else {
							log.info(REMOTE_IP
									+ "Successfully generated new RSA N");
							log.info(REMOTE_IP + "Now starting Fermat test");
							out.write(GilboaProtocol.NEW_N);
							out.flush();
						}
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "NEW_N");
					}
					break;

				case GilboaProtocol.N_FAILED:
					if (super.actualState.equals(states[7])
							|| super.actualState.equals(states[9])) {
						if (super.actualState.equals(states[9])) {
							log.info(REMOTE_IP
									+ "Other side found a witness that "
									+ "rsaN is not a product of 2 primes");
						} else {
							log
									.info(REMOTE_IP
											+ "Other side found that RSA n is not large enough");
						}
						super.actualState = states[3];
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "N_FAILED");
					}
					break;

				case GilboaProtocol.PREPARE_BIPRIME_TEST:
					log.debug(REMOTE_IP + "Got PREPARE_BIPRIME_TEST");
					if (super.actualState.equals(states[8])) {
						out.write(GilboaProtocol.PREPARE_BIPRIME_TEST);
						out.flush();
						readPrimeCertainty();
						primeTries = new Vector();
						super.actualState = states[9];
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL
								+ "PREPARE_BIPRIME_TEST");
					}
					break;

				case GilboaProtocol.BIPRIME_TEST:
					log.debug(REMOTE_IP + "Got BIPRIME_TEST");
					if (super.actualState.equals(states[9])) {
						doBiPrimeTest();
					} else {
						log.fatal(REMOTE_IP + PROTO_VIOL + "BIPRIME_TEST");
					}
					break;

				case GilboaProtocol.START_RSA_KEY_GENERATION:
					log.debug(REMOTE_IP + "Got START_RSA_KEY_GENERATION");
					if (super.actualState.equals(states[9])) {
						if (primeTries.size() == super.primeCertainty) {
							super.actualState = states[10];
							log
									.info("Verified that N is a product of two primes");
							log.info(REMOTE_IP + "Final rsaN is " + rsaN);
							log.debug(REMOTE_IP + "My p is " + p);
							log.debug(REMOTE_IP + "My q is " + q);
							log.info(REMOTE_IP + "Needed " + tries
									+ " tries to generate shared RSA N");
							out.write(GilboaProtocol.START_RSA_KEY_GENERATION);
							endThread();
						} else {
							log
									.fatal("remote side didn't do enough prime tests.");
							log.fatal(REMOTE_IP + "Exiting.");
							out.write(GilboaProtocol.VERIFICATION_FAILED);
							successful = false;
							endThread();
						}
					}
					out.flush();
					break;

				default:
					log.fatal(REMOTE_IP + "Unknown command. Exiting.");
					successful = false;
					endThread();
					break;

				}
			}
		} catch (final Exception e) {
			log.fatal(REMOTE_IP + "Caught Exception: ", e);
			successful = false;
		}
		log.debug(REMOTE_IP + "Last command was " + command);
		log.info(REMOTE_IP + "finished RSA N generation");
		synchronized (this) {
			notifyAll();
		}
	}

	/**
	 * Does a biPrime test for the RSA N. Notifies the client of the result and
	 * resets the server if an invalid rsaN is detected.
	 * 
	 * @throws Exception
	 *             If the protocol is not followed correctly.
	 * 
	 */
	private void doBiPrimeTest() throws Exception {
		BigInteger witness;

		// First, fix the shares of both parties without revealing them
		if (in.read() == GilboaProtocol.SHA1_DIGEST) {
			// Read in the other side's sha-1 digest
			byte[] remoteDigest = new byte[in.read()];
			in.readFully(remoteDigest);

			// Compute our own witness part
			witness = PrimeUtils.getRandom(rsaN, rand);

			// and inform the other party about it without revealing it
			byte[] wData = witness.toByteArray();
			SHA1Digest sha1 = new SHA1Digest();
			sha1.update(wData, 0, wData.length);
			byte[] digest = new byte[sha1.getDigestSize()];
			sha1.doFinal(digest, 0);
			out.write(GilboaProtocol.SHA1_DIGEST);
			out.write(digest.length);
			out.write(digest);
			out.flush();

			// Read the other parties candidate and verify it
			BigInteger remCand = (BigInteger) in.readObject();
			wData = remCand.toByteArray();
			sha1.reset();
			sha1.update(wData, 0, wData.length);
			sha1.doFinal(digest, 0);
			if ((new BigInteger(1, remoteDigest)).equals(new BigInteger(1,
					digest))) {

				// if it is correct, send our own candidate
				out.writeObject(witness);
				out.flush();

				// and add both candidates to get the final candidate
				witness = witness.add(remCand);

				// find the next needed value
				while (PrimeUtils.jacobiSymbol(witness, rsaN) != 1) {
					witness = witness.add(BigInteger.ONE);
				}
			} else {
				String msg = REMOTE_IP + PROTO_VIOL + "SHA-1 digest "
						+ "of transferred object is not correct";
				log.fatal(msg);
				throw new Exception(msg);
			}
		} else {
			String msg = REMOTE_IP + PROTO_VIOL + "expected SHA1_DIGEST";
			log.fatal(msg);
			throw new Exception(msg);
		}

		// Then do a simple test. If this test fails already, we do not have to
		// test in the twisted group
		if (in.read() == GilboaProtocol.WITNESS) {

			// verify that the other party found the same witness
			BigInteger compWitness = (BigInteger) in.readObject();
			if (compWitness.equals(witness)) {

				// read the other parties result
				final BigInteger remResult = (BigInteger) in.readObject();
				// compute our own result
				BigInteger jWitness = witness.modPow((rsaN.subtract(p)
						.subtract(q).add(BigInteger.ONE)).divide(BigInteger
						.valueOf(4)), rsaN);
				jWitness = jWitness.mod(rsaN);
				// and inform the client about our part
				out.write(GilboaProtocol.WITNESS);
				out.writeObject(jWitness);
				out.flush();

				// compute the final result
				jWitness = jWitness.multiply(remResult);
				jWitness = jWitness.mod(rsaN);
				if (jWitness.equals(BigInteger.ONE)
						|| jWitness.equals(rsaN.subtract(BigInteger.ONE))) {
					// if successful inform the client
					out.write(GilboaProtocol.N_VERIFIED);
					out.flush();
				} else {
					// if unsuccessful inform client
					log.info(REMOTE_IP + "Found witness that rsaN"
							+ " is not a product of two primes.");
					out.write(GilboaProtocol.N_FAILED);
					out.flush();

					// and reset state machine so that a new N must be generated
					super.actualState = states[3];
					primeTries.clear();
					return;
				}
			} else {
				String msg = REMOTE_IP + PROTO_VIOL
						+ "Not using the same witness";
				log.fatal(msg);
				throw new Exception(msg);
			}
		}

		// The previous test was successful. Now do a test in the twisted group.

		BigInteger twWitness[] = PrimeUtils.getRandomTwistedElement(rsaN, rand);
		if (in.read() == GilboaProtocol.SHA1_DIGEST) {
			// read the sha-1 digest of the client's candidate
			byte[] remoteDigest = new byte[in.read()];
			in.readFully(remoteDigest);
			// compute our own digest
			byte[] wData = twWitness[0].toByteArray();
			SHA1Digest sha1 = new SHA1Digest();
			sha1.update(wData, 0, wData.length);
			wData = twWitness[1].toByteArray();
			sha1.update(wData, 0, wData.length);
			byte[] digest = new byte[sha1.getDigestSize()];
			sha1.doFinal(digest, 0);

			// and inform the client about it
			out.write(GilboaProtocol.SHA1_DIGEST);
			out.write(digest.length);
			out.write(digest);
			out.flush();

			// read in the client's candidate
			BigInteger remWitness[] = new BigInteger[2];
			remWitness[0] = (BigInteger) in.readObject();
			remWitness[1] = (BigInteger) in.readObject();
			// and verify its digest
			sha1.reset();
			wData = remWitness[0].toByteArray();
			sha1.update(wData, 0, wData.length);
			wData = remWitness[1].toByteArray();
			sha1.update(wData, 0, wData.length);
			sha1.doFinal(digest, 0);
			if ((new BigInteger(1, remoteDigest)).equals(new BigInteger(1,
					digest))) {
				// if digest is correct, reveal our candidate
				out.writeObject(twWitness[0]);
				out.writeObject(twWitness[1]);
				out.flush();
				// and compute final composed candidate
				twWitness = PrimeUtils.twistedAdd(remWitness, twWitness);
			} else {
				// else abort computation
				String msg = REMOTE_IP + PROTO_VIOL + "SHA-1 digest "
						+ "of transferred object is not correct";
				log.fatal(msg);
				throw new Exception(msg);
			}
		} else {
			String msg = REMOTE_IP + PROTO_VIOL + "expected SHA1_DIGEST";
			log.fatal(msg);
			throw new Exception(msg);
		}

		// read the client's composed candidate
		final BigInteger remTwWitnessResult[] = new BigInteger[2];
		if (in.read() == GilboaProtocol.TW_WITNESS) {
			final BigInteger remWitness[] = new BigInteger[2];
			remWitness[0] = (BigInteger) in.readObject();
			remWitness[1] = (BigInteger) in.readObject();
			if (!remWitness[0].equals(twWitness[0])
					|| !remWitness[1].equals(twWitness[1])) {
				// if they are not equal, the other side is cheating, exit
				throw new Exception("witnesses not equal");
			}
			// read the client's part of the test
			remTwWitnessResult[0] = (BigInteger) in.readObject();
			remTwWitnessResult[1] = (BigInteger) in.readObject();
		} else {
			final String msg = PROTO_VIOL + "expecting TW_WITNESS";
			log.fatal(REMOTE_IP + msg);
			throw new Exception(msg);
		}

		// compute our part of the test
		BigInteger[] jTwWitness = PrimeUtils.twistedModPow(twWitness, rsaN.add(
				p).add(q).add(BigInteger.ONE), rsaN);
		// and inform the client about it
		out.write(GilboaProtocol.TW_WITNESS);
		out.writeObject(jTwWitness[0]);
		out.writeObject(jTwWitness[1]);
		out.flush();

		// compute the final outcome
		jTwWitness = PrimeUtils.twistedModMult(jTwWitness, remTwWitnessResult,
				rsaN);

		if (jTwWitness[1].equals(BigInteger.ZERO)) {
			// if successful, inform the client
			log.info(REMOTE_IP + "For witness " + witness
					+ ", the rsaN passes the test");
			out.write(GilboaProtocol.N_VERIFIED);
			out.flush();
			// hopefully it thinks the same ;)
			if (in.read() == GilboaProtocol.N_VERIFIED) {
				log.info(REMOTE_IP + "RSA n verified for " + witness);
				// Record the successful test in our statistics, so we can
				// verify later that we did enough tests
				primeTries.add(witness);
			} else {
				// if parties disagree, exit
				final String msg = "Other side thinks that " + rsaN
						+ " is not prime, but I do";
				log.fatal(REMOTE_IP + msg);
				throw new Exception(msg);
			}
		} else {
			// if unsuccessful, also inform the client
			log.info(REMOTE_IP + "Found witness "
					+ "that rsaN is not a product of two primes.");
			out.write(GilboaProtocol.N_FAILED);
			out.flush();

			// and reset state machine
			super.actualState = states[3];
			primeTries.clear();
		}
	}

	/**
	 * @throws Exception
	 * 
	 */
	private void readPrimeCertainty() throws Exception {
		if (in.read() == GilboaProtocol.PRIME_CERTAINTY) {
			final int remPrimeCertainty = in.readInt();
			super.primeCertainty = Math.max(super.primeCertainty,
					remPrimeCertainty);
			out.write(GilboaProtocol.PRIME_CERTAINTY);
			out.writeInt(super.primeCertainty);
			out.flush();
		} else {
			throw new Exception(PROTO_VIOL + "PRIME_CERTAINTY");
		}
	}

	/**
	 * @return
	 */
	private boolean verifyGeneration() {
		for (int i = 0; i < computedZs.size(); i++) {
			if (!(computedZs.get(i) instanceof BigInteger)) {
				return false;
			}
		}
		p = PrimeUtils.chineseRemainder(pComposites, genPrimes);
		q = PrimeUtils.chineseRemainder(qComposites, genPrimes);
		log.debug(REMOTE_IP + "My generated p is " + p + " with "
				+ p.bitLength() + " bit.");
		log.debug(REMOTE_IP + "My generated q is " + q + " with "
				+ q.bitLength() + " bit.");
		return true;
	}

	/**
	 * 
	 */
	private void prepareZVector() {
		final Object o = new Object();
		while (computedZs.size() < genPrimes.size()) {
			computedZs.add(o);
			pComposites.add(o);
			qComposites.add(o);
		}
		for (int i = 0; i < computedZs.size(); i++) {
			computedZs.set(i, o);
			pComposites.set(i, o);
			qComposites.set(i, o);
		}
		// Make sure that n mod 4 is correct
		int pos = genPrimes.indexOf(four);
		pComposites.set(pos, nMod4);
		qComposites.set(pos, nMod4);
		// 3*3 is 1 mod 4 ;)
		computedZs.set(pos, BigInteger.ONE);
	}

	/**
	 * Computes the minimum of servers and clients preferred key size and writes
	 * it to the client. Now both parties talk about the same keySize and should
	 * use the same primes. This is verified by checkCandidateTable later in the
	 * protocol.
	 * 
	 * @throws IOException
	 */
	private void agreeOnPrimeLength() throws IOException {
		log.debug(REMOTE_IP + "My preferred key size: "
				+ EnvironmentSetup.getDesiredKeySize());
		final int remoteKeySize = in.readInt();
		log.debug(REMOTE_IP + "Remote preferred key size: " + remoteKeySize);
		preferredSharedKeySize = Math.min(EnvironmentSetup.getDesiredKeySize(),
				remoteKeySize);
		log
				.debug(REMOTE_IP + "preferred Key Size is "
						+ preferredSharedKeySize);
		log.debug(REMOTE_IP + "Writing PREFERRED_SHARED_KEY_SIZE");
		out.write(GilboaProtocol.PREFERRED_SHARED_KEY_SIZE);
		out.writeInt(preferredSharedKeySize);
		out.flush();
	}

	/**
	 * Computes N' = p_c*q_s + q_c*p_s + p_s*q_s mod <tt>prime</tt>. In the
	 * Gilboa paper these are named z1, z2 and z3, N' is named z.
	 * 
	 * @param prime
	 *            The prime for that N' shall be computed
	 * @throws Exception
	 *             If the protocol is violated.
	 */
	private void computeN_(final BigInteger prime) throws Exception {
		// get the other parties public benaloh system
		final BenalohSystem bs = (BenalohSystem) benalohSystems.get(prime);
		final BenalohEngine pubEngine = bs.getPubEngine();

		// read in clients p
		byte[] p_c;
		if (in.read() == GilboaProtocol.REMOTE_Z1) {
			final int size = in.readInt();
			p_c = new byte[size];
			in.readFully(p_c);
		} else {
			throw new Exception(PROTO_VIOL + "REMOTE_Z1");
		}

		// read in clients q
		byte[] q_c;
		if (in.read() == GilboaProtocol.REMOTE_Z2) {
			final int size = in.readInt();
			q_c = new byte[size];
			in.readFully(q_c);
		} else {
			throw new Exception(PROTO_VIOL + "REMOTE_Z2");
		}

		// set own sharings of p and q mod prime
		BigInteger pModPrime;
		BigInteger qModPrime;
		if (super.actualState.equals(states[4])) {
			// while generating p and q
			final int pos = genPrimes.indexOf(prime);
			BigInteger[] pq = generateNewPrimes(prime);
			// set p and q in the lists used to reconstruct them via CRT
			pComposites.set(pos, pq[0]);
			qComposites.set(pos, pq[1]);
			pModPrime = pq[0];
			qModPrime = pq[1];
		} else {
			// p and q are fixed, set correct value
			pModPrime = p.mod(prime);
			qModPrime = q.mod(prime);
		}

		// exponentiate p_c with pModPrime, this is equivalent to a plaintext
		// multiplication
		p_c = pubEngine.multiplyCryptedBlock(p_c, qModPrime);
		// exponentiate q_c with pModPrime, this is equivalent to a plaintext
		// multiplication
		q_c = pubEngine.multiplyCryptedBlock(q_c, pModPrime);

		// Compute p*q mod prime
		final byte[] pqModPrime = pModPrime.multiply(qModPrime).mod(prime)
				.toByteArray();

		// encrypt it
		byte[] n_ = pubEngine.processBlock(pqModPrime, 0, pqModPrime.length);

		// multiply all crypted blocks successively, this is equivalent to
		// addition in plaintext
		n_ = pubEngine.addCryptedBlocks(n_, q_c);
		n_ = pubEngine.addCryptedBlocks(n_, p_c);

		// inform the client about the newly computed N'
		out.write(GilboaProtocol.COMPUTED_N_);
		out.writeObject(prime);
		out.writeInt(n_.length);
		out.write(n_);
		out.flush();

		// check result of complete computation
		final int command = in.read();
		if (command == GilboaProtocol.COMPUTED_N_) {
			// everything went well, p*q mod prime != 0
			if (super.actualState.equals(states[4])) {
				computedZs.set(genPrimes.indexOf(prime), prime);
			}
			return;
		} else if ((command == GilboaProtocol.N_MOD_PRIME_IS_0)
				&& super.actualState.equals(states[4])) {
			// N mod prime is 0, needs to be redone
			log.info(REMOTE_IP + "Got N_MOD_PRIME_IS_0 for prime " + prime);
		} else if (command == GilboaProtocol.N_FAILED) {
			// N failed finally and cannot be adjusted. Restart needed
			log.info(REMOTE_IP + "Restarting generation, rsaN failed for "
					+ prime);
			// reset state machine to correct state
			super.actualState = states[3];
		} else {
			// got protocol violation, exit
			final String msg = PROTO_VIOL + "Wrong command: " + command;
			log.debug(REMOTE_IP + "actual state is " + super.actualState);
			log.fatal(REMOTE_IP + msg);
			throw new Exception(msg);
		}
	}

	private void readBenalohSystem() throws IOException, ClassNotFoundException {
		// log.debug(REMOTE_IP + "reading remote public key");
		final BigInteger y = (BigInteger) in.readObject();
		final BigInteger r = (BigInteger) in.readObject();
		final BigInteger n = (BigInteger) in.readObject();
		final BenalohKeyParameters key = new BenalohKeyParameters(y, r, n);
		super.benalohSystems.put(r, new BenalohSystem(key));
		benProd = benProd.multiply(r);
	}

}
