/**
 * 
 */
package net.sf.distrib_rsa.protocols.straubSharing;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.net.ssl.SSLSocket;

import net.sf.distrib_rsa.EnvironmentSetup;
import net.sf.distrib_rsa.cryptosystems.PrimeUtils;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternEngine;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeyParameters;
import net.sf.distrib_rsa.cryptosystems.naccacheStern.NaccacheSternKeySerializationFactory;
import net.sf.distrib_rsa.protocols.RSAParameters;

import org.apache.log4j.Logger;

/**
 * @author lippold
 * 
 */
public abstract class StraubCommon extends Thread implements RSAParameters {

	final private static Logger log = Logger
			.getLogger(StraubCommon.class);

	final protected static String[] states = { "EXCHANGE_KEY_SIZE",
			"EXCHANGE_NCS_KEYS", "CANDIDATE_GENERATION", "CANDIDATE_FOUND",
			"CANDIDATE_VERIFICATION", "CANDIDATE_VERIFIED", "PUBLISH_N_PART",
			"CONVERT Q1 P2", "CONVERT Q1 P2 FINISHED", "A_MOD_2",
			"CONVERT (P1 + Q1)P2", "CONVERT (P1+Q1)P2 FINISHED", "B_MOD_2",
			"FERMAT_TEST" };

	/**
	 * @uml.property  name="actualState"
	 */
	protected String actualState;

	final protected static String PROTO_VIOL = "Protocol violation. Not executing ";

	/**
	 * @uml.property  name="out"
	 */
	final protected ObjectOutputStream out;

	/**
	 * @uml.property  name="in"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="java.math.BigInteger"
	 */
	final protected ObjectInputStream in;

	/**
	 * @uml.property  name="running"
	 */
	protected boolean running = true;

	/**
	 * @uml.property  name="preferredSharedKeySize"
	 */
	protected int preferredSharedKeySize = EnvironmentSetup.getDesiredKeySize();

	/**
	 * @uml.property  name="generatingPrimes"
	 * @uml.associationEnd  qualifier="new:java.math.BigInteger de.uni_bremen.informatik.lippold.protocols.straubSharing.AddMultRemainders"
	 */
	protected Hashtable generatingPrimes = new Hashtable();

	/**
	 * @uml.property  name="verifyingPrimes"
	 * @uml.associationEnd  qualifier="prime:java.math.BigInteger de.uni_bremen.informatik.lippold.protocols.straubSharing.AddMultRemainders"
	 */
	protected Hashtable verifyingPrimes = new Hashtable();

	/**
	 * @uml.property  name="multToAddPrimes"
	 * @uml.associationEnd  qualifier="valueOf:java.math.BigInteger de.uni_bremen.informatik.lippold.protocols.straubSharing.AddMultRemainders"
	 */
	protected Hashtable multToAddPrimes = new Hashtable();

	/**
	 * @uml.property  name="q"
	 */
	final protected BigInteger q;

	/**
	 * @uml.property  name="pubEngine"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	final protected NaccacheSternEngine pubEngine;

	/**
	 * @uml.property  name="privEngine"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	final protected NaccacheSternEngine privEngine;

	/**
	 * @uml.property  name="remotePubEngine"
	 * @uml.associationEnd  
	 */
	protected NaccacheSternEngine remotePubEngine;

	/**
	 * @uml.property  name="rand"
	 */
	final protected SecureRandom rand;

	final private static int PROCESSORS = Runtime.getRuntime()
			.availableProcessors();

	/**
	 * @uml.property  name="remSigma"
	 */
	protected BigInteger remSigma;

	/**
	 * @uml.property  name="candidate"
	 * @uml.associationEnd  
	 */
	protected AddMultRemainders candidate;

	/**
	 * @uml.property  name="rsaN"
	 */
	protected BigInteger rsaN;

	/**
	 * @uml.property  name="myMQ"
	 */
	protected BigInteger myMQ;

	/**
	 * @uml.property  name="myA"
	 */
	protected BigInteger myA;

	/**
	 * @uml.property  name="myB"
	 */
	protected BigInteger myB;

	/**
	 * @uml.property  name="genProd"
	 */
	protected BigInteger genProd;

	/**
	 * @uml.property  name="verProd"
	 */
	protected BigInteger verProd;

	/**
	 * @uml.property  name="mult2addProd"
	 */
	protected BigInteger mult2addProd;

	/**
	 * @uml.property  name="remoteID"
	 */
	protected final String remoteID;

	/**
	 * @uml.property  name="mySigma"
	 */
	protected final BigInteger mySigma = EnvironmentSetup.getNcsPubKey()
			.getSigma();

	/**
	 * @uml.property  name="successful"
	 */
	protected boolean successful = true;
	
	/**
	 * Superclass for distributed Sieving
	 * 
	 * @param socket
	 *            the socket this thread communicates over
	 * @throws IOException
	 *             if the streams cannot be constructed from the socket
	 * @throws NoSuchProviderException
	 * @throws NoSuchAlgorithmException
	 */
	public StraubCommon(final SSLSocket socket) throws IOException,
			NoSuchAlgorithmException, NoSuchProviderException {
		remoteID = "IP: " + socket.getInetAddress() + "\nCertificate: "
				+ socket.getSession().getPeerCertificates()[0];
		out = new ObjectOutputStream(socket.getOutputStream());
		in = new ObjectInputStream(socket.getInputStream());
		pubEngine = new NaccacheSternEngine(PROCESSORS);
		pubEngine.init(true, EnvironmentSetup.getNcsPubKey());
		privEngine = new NaccacheSternEngine(PROCESSORS);
		privEngine.init(false, EnvironmentSetup.getNcsPrivKey());
		rand = EnvironmentSetup.getSecureRandom();
		// In [Straub04] it is recommended to take p_x.bitLength and
		// q_x.bitLength
		// to preferredSharedKeySize/2.
		q = new BigInteger(EnvironmentSetup.getDesiredKeySize() / 2,
				EnvironmentSetup.getPrimeCertainty(), rand);
	}

	public void endThread() {
		running = false;
	}

	protected void populatePrimeLists() throws NoSuchAlgorithmException,
			NoSuchProviderException {
		log.debug("populating prime lists");
		// We build our RSA-Modulus out of three Primes (p_1 + p_2) q_1 q_2
		// One party knows p_1 and q_1, the other p_2 and q_2.

		// fetch the first n primes
		final int[] minPrimes = PrimeUtils.getFirstPrimes(preferredSharedKeySize);

		int primeCnt = 0;
		genProd = BigInteger.ONE;
		final Object o = new Object();
		// From [Straub04]: P (here genProd) is the minimal set of the smallest
		// primes, so that P exceeds 2^(l-1). (l is bitlength of the shared key)
		while (genProd.bitLength() <= preferredSharedKeySize) {
			final BigInteger prime = BigInteger.valueOf(minPrimes[primeCnt]);
			genProd = genProd.multiply(prime);
			generatingPrimes.put(prime, o);
			primeCnt++;
		}
		log.debug("P has " + generatingPrimes.size()
				+ " primes, the largest prime is " + minPrimes[primeCnt - 1]);

		verProd = BigInteger.ONE;
		while (verProd.bitLength() <= (preferredSharedKeySize + 2 + 1)) {
			final BigInteger prime = BigInteger.valueOf(minPrimes[primeCnt]);
			verProd = verProd.multiply(prime);
			verifyingPrimes.put(prime, o);
			primeCnt++;
		}
		log.debug("P' has " + verifyingPrimes.size()
				+ " primes, the largest prime is " + minPrimes[primeCnt - 1]);

		primeCnt = generatingPrimes.size() - 1;
		final BigInteger twoPowMu = BigInteger.valueOf(2).pow(
				preferredSharedKeySize / 2);
		final BigInteger shareProd = (verProd.add(twoPowMu)).multiply(twoPowMu);
		multToAddPrimes.putAll(generatingPrimes);
		multToAddPrimes.remove(BigInteger.valueOf(2));
		mult2addProd = genProd.divide(BigInteger.valueOf(2));
		while (shareProd.compareTo(mult2addProd) > 0) {
			final BigInteger prime = BigInteger.valueOf(minPrimes[primeCnt]);
			mult2addProd = mult2addProd.multiply(prime);
			multToAddPrimes.put(prime, o);
			primeCnt++;
		}
		log.debug("P'' has " + multToAddPrimes.size()
				+ " primes, the largest prime is " + minPrimes[primeCnt - 1]);

	}

	protected void sendNCSPubKey() throws IOException {
		log.debug("Writing REMOTE_NCS_PUB_KEY");
		out.write(StraubProtocol.REMOTE_NCS_PUB_KEY);
		final byte[] serialized = NaccacheSternKeySerializationFactory
				.getSerialized(EnvironmentSetup.getNcsPubKey());
		out.writeInt(serialized.length);
		out.write(serialized);
		out.flush();
	}

	protected void readRemotePubKey() throws IOException,
			ClassNotFoundException {
		log.debug("reading remote public key");
		final int length = in.readInt();
		final byte[] serializedKey = new byte[length];
		in.read(serializedKey);
		final NaccacheSternKeyParameters remotePubKey = NaccacheSternKeySerializationFactory
				.deserialize(serializedKey);
		remSigma = remotePubKey.getSigma();
		remotePubEngine = new NaccacheSternEngine(PROCESSORS);
		remotePubEngine.init(true, remotePubKey);
		log.debug("Remote Public Key engine initialized");
	}

	/**
	 * Converts a [ prime, AddMultRemainders ] table to a AddMultRemainder with
	 * CRT.
	 * 
	 * @param table
	 *            the table to be converted.
	 * @return the AddMultRemainder that has the value a % p_i = a_i, m % p_1 =
	 *         m_i.
	 */
	protected AddMultRemainders crtCandidate(final Hashtable table) {
		log.debug("generating candidate from remainders");
		final Vector primes = new Vector();
		final Vector addRemainders = new Vector();
		final Vector multRemainders = new Vector();
		final Enumeration keys = table.keys();
		while (keys.hasMoreElements()) {
			final BigInteger prime = (BigInteger) keys.nextElement();
			primes.add(prime);
			final AddMultRemainders amr = (AddMultRemainders) table.get(prime);
			addRemainders.add(amr.getAdditive());
			multRemainders.add(amr.getMultiplicative());
		}
		return new AddMultRemainders(PrimeUtils.chineseRemainder(addRemainders, primes),
				PrimeUtils.chineseRemainder(multRemainders, primes), remSigma);
	}

	protected void fermatTest() {
		// TODO
	}


	protected void readRSAModulus() throws IOException, ClassNotFoundException {
		final BigInteger remoteMQ = (BigInteger) in.readObject();
		rsaN = remoteMQ.multiply(myMQ);
		log.info("RSA N for Fermat-Testing: " + rsaN);
	}

	protected void resetMultToAddPrimes() {
		final Object o = new Object();
		final Enumeration en = multToAddPrimes.keys();
		final Hashtable newHT = new Hashtable();
		while (en.hasMoreElements()) {
			newHT.put(en.nextElement(), o);
		}
		multToAddPrimes = newHT;
	}
	
	public BigInteger getP(){
		//TODO
		return null;
	}
	public BigInteger getQ(){
		// TODO
		return null;
	}
	public BigInteger getRsaN(){
		// TODO
		return null;
	}

	/**
	 * @return the successful
	 */
	public boolean wasSuccessful() {
		return successful;
	}
}
